import {Component, Injector, OnInit, TemplateRef, ViewChild} from "@angular/core";
import {BaseComponent} from "../../../../../system/controllers/BaseComponent";
import {IsbnApplicationService} from "../../../../../shared/services/isbn-application.service";
import {
  ActionFlyMenuOptionsInterface,
  FlyMenuActionEnum,
} from "../../../../../system/components/action-fly-menu/action-fly-menu.component";
import {BaseResourceInterface} from "../../../../../system/datamodels/BaseResourceInterface";
import {IsbnIssueModalComponent} from "../../../../../shared/isbn-issue-modal/isbn-issue-modal.component";
import {environment} from "../../../../../../environments/environment";
import {BaseIssuedISBN} from "../../../../../shared/base/contituents/isbn-application/models/BaseIssuedISBN";
import {BaseIsbnApplication} from "../../../../../shared/base/contituents/isbn-application/models/BaseIsbnApplication";
import {AdvancedSelectConfigInterface} from "../../../../../system/components/advanced-select/advanced-select.component";
import {AuthorsListComponent} from "../../../../../shared/authors-list-component/authors-list.component";
import {Observable, Subject} from "rxjs";

@Component({
  selector: 'app-isbn-application-list',
  templateUrl: './isbn-application-list.component.html'
})
export class IsbnApplicationListComponent extends BaseComponent implements  OnInit {

  @ViewChild('publisherSelectItemTemplate') publisherSelectItemTemplate: TemplateRef<any>;
  @ViewChild('emptySelectionTemplate') publisherEmptyItemTemplate: TemplateRef<any>;

  /**
   * @var config
   */
  public config: AdvancedSelectConfigInterface = {};

  public publisherModel;

  public receivedModel = null;

  public publisherFetchUrl = "";

  constructor (injector: Injector) {
    super(injector);
    this.service = injector.get(IsbnApplicationService);
  }

  protected setAdvSelectConfig() {
    this.config.itemTemplate = this.publisherSelectItemTemplate;
    this.config.equalityCheckProperty = 'id';
    this.config.actionButtonEnabled = false;
    this.config.disableEmptyField = false;
    this.config.selectInfoLabel = "Select publisher";
    this.config.emptySelectionTemplate = this.publisherEmptyItemTemplate;
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.pageTitle.setTitle("ISBN applications");
    this.getList();
    this.publisherFetchUrl = this.publisherFetchUrl = this.service.getBaseApiUrl() + '/publishers';
    this.setAdvSelectConfig();
  }
  public getList(showToast?: boolean): Observable<any> {
    const subject = new Subject();
    super.getList().subscribe((items: any) => {

      for (let i = 0; i < items.length; i++ ) {
        const item = items.getItem(i);
        const flyMenuAddOn = [];
        flyMenuAddOn.push({
          name: 'show_more', label: 'Show more...', active: true
        });
        // tslint:disable-next-line:triple-equals
        if (item.received_status != 1 && this.authUser.can('isbn_applications.mark_received')) {
          flyMenuAddOn.push ({name: 'mark_as_received', label: 'Mark as received', active: true, });
        }
        if (!item.issued_isbn && this.authUser.can('isbn.create')) {
          flyMenuAddOn.push ({name: 'issue_isbn', label: 'Issue ISBN', active: true, });
        }

        item.flyMenuAddOn = flyMenuAddOn;
        item.flyMenuConfig =  {
          disableEditItem: !this.authUser.can('isbn_applications.update') || item.issued_isbn,
          disableRemoveItem: true,
          disableStatusItem: true
        };

      }
    });
    return subject;
  }

  public onPublisherChange () {
    if (this.publisherModel) {
      this.urlParams.publisher = this.publisherModel.id;
    } else {
      delete this.urlParams.publisher;
    }
    this.getList();
  }
  public onReceivedModelChange () {
    if (isNaN(this.receivedModel)) {
      delete this.urlParams.received;
    } else {
      this.urlParams.received = this.receivedModel;
    }
    this.getList();
  }

  /**
   * Edit the item
   * @param item
   */
  public editItem(item: BaseResourceInterface) {
    this.router.navigate([`manage/isbn-applications/${item.id}/edit`]);
  }

  public showIsbnModalBox (application: BaseIsbnApplication, isbn?: BaseIssuedISBN) {
    this.nicoCtrl.create(IsbnIssueModalComponent, {
      htmlTemplateAsIs: true,
      data: {application: application, isbn: isbn, isbnAddOn: environment.country_isbn_identifier},
      onOkayProcessComplete: ( model: any, message: any) => {
        this.showSuccessToast(message);
        this.getList();
      }
    }).present();
  }

  public onFlyMenuAction(evt: FlyMenuActionEnum | any, item: any) {
    super.onFlyMenuAction(evt, item);
    if (evt === 'mark_as_received') {
      const subscription = (<IsbnApplicationService> this.service).markAsReceived(item).subscribe(() => {
        if (subscription) {
          subscription.unsubscribe();
        }
        this.items.forEach((itm: any) => {
          if (itm.id === item.id) {
            item.received_status = 1;
          }
        });
      }, ( ) => {
        if (subscription) {
          subscription.unsubscribe();
        }
      });
    } else if (evt === 'issue_isbn') {
      this.showIsbnModalBox (item);
    } else if (evt === 'show_more') {
      this.router.navigate(['manage/isbn-applications/' + item.id + '/show']);
    }
  }

  public editIsbn(application: BaseIsbnApplication, isbn: BaseIssuedISBN ) {
    this.showIsbnModalBox(application, isbn);
  }

  public authorListShowMore (item: any) {
    this.nicoCtrl.create(AuthorsListComponent, {
      data: {items: item.authors},
      htmlTemplateAsIs: true
    }).present();
  }


}
