import {Component, Injector, OnInit, TemplateRef, ViewChild} from "@angular/core";
import {BaseComponent} from "../../../../../system/controllers/BaseComponent";
import {FormArray, FormGroup, Validators} from "@angular/forms";
import {IsbnTypeEnum} from "../../../../../shared/base/contituents/isbn-application/enums/isbn.type.enum";
import {IsbnIllustrationTypeEnum} from "../../../../../shared/base/contituents/isbn-application/enums/isbn-illustration-type.enum";
import {IsbnIllustrationColorEnum} from "../../../../../shared/base/contituents/isbn-application/enums/isbn-illustration-color.enum";
import {IsbnApplicationService} from "../../../../../shared/services/isbn-application.service";
import {Observable, Subject} from "rxjs";
import * as moment from "moment";
import {AdvancedSelectConfigInterface} from "../../../../../system/components/advanced-select/advanced-select.component";
import {BasePublisher} from "../../../../../shared/base/contituents/publisher/models/BasePublisher";
import {IsbnApplication} from "../../models/IsbnApplication";
import {ServerResponse} from "../../../../../system/responses/ServerResponse";
import {UploadFileOptionsInterface} from "../../../../../shared/upload-file-component/upload-file.options.interface";
import {Collection} from "../../../../../system/utilities/Collection";
import {AppLanguageProviderService} from "../../../../../shared/services/app-language-provider.service";
import { IDropdownSettings } from 'ng-multiselect-dropdown';
import {IsbnIssueModalComponent} from "../../../../../shared/isbn-issue-modal/isbn-issue-modal.component";
import {BaseIsbnApplication} from "../../../../../shared/base/contituents/isbn-application/models/BaseIsbnApplication";
import {BaseIssuedISBN} from "../../../../../shared/base/contituents/isbn-application/models/BaseIssuedISBN";
import { Helper } from "src/app/system/utilities/Helper";

@Component({
  selector: 'app-isbn-application-create',
  templateUrl: './isbn-application-create.component.html'
})
export class IsbnApplicationCreateComponent extends BaseComponent implements OnInit {

  public formGroup: FormGroup;

  public formId = "isbn-application-create-form";

  public publisherFetchUrl: string;

  public moduleTitle = "Create new application";

  public currentDate = null;

  public calenderModle = null;

  public freeOfCostModel = null;

  public freeOfCost:string = "Free Of Cost";

  dropdownList = [];
  selectedItems = [];
  dropdownSettings:IDropdownSettings = {};

  public haveApplicationId = this.authUser.can('isbn_applications.make_application_id');
  public accessionNumber = this.authUser.can('isbn_applications.accessionNumber');
  public callNumber = this.authUser.can('isbn_applications.callNumber');
  public bookNumber = this.authUser.can('isbn_applications.bookNumber');
  public keywords = this.authUser.can('isbn_applications.keywords');


  @ViewChild('publisherSelectItemTemplate') publisherSelectItemTemplate: TemplateRef<any>;

  /**
   * @var config
   */
  public config: AdvancedSelectConfigInterface = {};

  public isbnTypeOptions: { value: IsbnTypeEnum, label: string }[] = [
    {value: IsbnTypeEnum.Book, label: 'Book'},
    {value: IsbnTypeEnum.CassetteCD, label: 'Cassette or CD'},
    {value: IsbnTypeEnum.EBook, label: 'E-book'},
    {value: IsbnTypeEnum.ComputerSoftware, label: 'computer software'},
    {value: IsbnTypeEnum.Map, label: 'Map'},
    {value: IsbnTypeEnum.MicroFilm, label: 'Microfilm / Microfiche'},
    {value: IsbnTypeEnum.Others, label: 'Other'},
  ];

  public illustrationTypeOptions: { item_id: IsbnIllustrationTypeEnum, item_text: string }[]=[
    {item_id: IsbnIllustrationTypeEnum.Chart, item_text: 'Chart'},
    {item_id: IsbnIllustrationTypeEnum.Map, item_text: 'Map'},
    {item_id: IsbnIllustrationTypeEnum.Tables, item_text: 'Tables'},
    {item_id: IsbnIllustrationTypeEnum.None, item_text: 'None'},

  ];

  public illustrationColorOptions: { value: IsbnIllustrationColorEnum, label: string }[] = [
    {value: IsbnIllustrationColorEnum.BlackAndWhite, label: 'Black and white'},
    {value: IsbnIllustrationColorEnum.Color, label: 'Color'},
    {value: IsbnIllustrationColorEnum.None, label: 'None'},
  ];


  public currencyOptions: { value: string, label: string }[] = [
    {value: 'Nepali (Rs.)', label: 'Nepali (Rs.)'},
    {value: 'Indian rupee (INR)', label: 'Indian rupee (INR)'},
    {value: 'Dollor ($)', label: 'Dollor ($)'},
    {value: 'Australian Dollor', label: 'Australian Dollor ($)'},
    {value: 'Pound sterling', label: 'Pound sterling (£)'},
    {value: 'Euro', label: 'Euro (€)'},
    {value: 'Canadian Dollar', label: 'Canadian Dollar ($)'},
    {value: 'Japanese yen', label: 'Japanese yen (¥)'},
    {value: 'Chinese Yuan', label: 'Chinese Yuan (¥);'}
  ];

  public coverType: string[] = ['paperback', 'hardbound', 'other'];

  public languageOptions: Collection<any> = new Collection<any>();

  protected languageService: AppLanguageProviderService;

  public editionOptions: string[] = ['First', 'Second', 'Third', 'Fourth', 'Fifth', 'Sixth',
    'Seventh', 'Eighth', 'Ninth', 'Tenth', 'Eleventh', 'Twelfth', 'Thirteenth', 'Fourteenth', 'Fifteenth', 'Sixteenth',
    'Seventeenth', 'Eighteenth', 'Nineteenth', 'Twentieth', 'Twenty first', 'Twenty second', 'Twenty third', 'Twenty fourth',
    'Twenty fifth', 'Twenty sixth', 'Twenty seventh', 'Twenty eighth', 'Twenty ninth', 'Thirtieth', 'Thirty first',
    'Thirty second', 'Thirty third', 'Thirty fourth', 'Thirty fifth', 'Thirty sixth', 'Thirty seventh', 'Thirty eighth', 'Thirty ninth', 'fortieth', 'Forty first', 'Forty second', 'Forty third', 'Forty fourth', 'Forty fifth', 'Forty sixth', 'Forty seventh', 'Forty eighth', 'Forty ninth', 'Fiftieth', 
    'Fifty first', 'Fifty second', 'Fifty third', 'Fifty fourth', 'Fifty fifth', 'Fifty sixth', 'Fifty seventh', 'Fifty eighth', 'Fifty ninth', 'Sixtieth', 
    'Sixty first', 'Sixty second', 'Sixty third', 'Sixty fourth', 'Sixty fifth', 'Sixty sixth', 'Sixty seventh', 'Sixty eighth', 'Sixty ninth', 'Seventieth', 
    'Seventy first', 'Seventy second', 'Seventy third', 'Seventy fourth', 'Seventy fifth', 'Seventy sixth', 'Seventy seventh', 'Seventy eighth', 'Seventy ninth', 'Eightieth',
    'Eighty first', 'Eighty second', 'Eighty third', 'Eighty fourth', 'Eighty fifth', 'Eighty sixth', 'Eighty seventh', 'Eighty eighth', 'Eighty ninth', 'Ninetieth',
    'Ninety first', 'Ninety second', 'Ninety third', 'Ninety fourth', 'Ninety fifth', 'Ninety sixth', 'Ninety seventh', 'Ninety eighth', 'Ninety ninth', 'Hundredth'
  ];

  public coverUploadOption: UploadFileOptionsInterface;

  public paymentSlipUploadOption: UploadFileOptionsInterface;

  constructor(injector: Injector) {
    super(injector);
    this.service = injector.get(IsbnApplicationService);
    this.languageService = injector.get(AppLanguageProviderService);
    this.publisherFetchUrl = this.service.getBaseApiUrl() + '/publishers';
    this.calenderModle = this.currentDate = moment().format('YYYY-MM-DD');    
  }

  protected setUploadOptions(forId: string, label: string) {
    const opt: UploadFileOptionsInterface = {
      title: label,
      instantUpload: true,
      cancellable: false,
      disabledOnComplete: true,
      inputId: forId,
      hidden: false,
      uploadInputConfig: {
        type: 'uploadAll',
        url: this.service.getBaseApiUrl() + '/upload-resources',
        headers: {'Authorization': 'Bearer ' + this.session.getToken()},
        method: 'POST',
        fieldName: 'file',
        data: {'type': 'image'},
      }
    };
    if (forId === 'book_cover_url') {
      this.coverUploadOption = opt;
    } else if (forId === 'deposit_slip_url') {
      this.paymentSlipUploadOption = opt;
    }
  }

  protected setModelIfEditPage() {
    if (this.routeParams.get('id')) {
      const spinner = this.ajaxSpinner.showSpinner();
      this.service.find(this.routeParams.get('id'))
        .subscribe((model: IsbnApplication) => {
          spinner.hide();
          this.model = model;
          // create form array for number of authors
          if (this.model.authors) {
            this.removeAuthor(0, true);
            for (let i = 0; i < this.model.authors.length; i++ ) {
              this.addAuthor();
            }
          } 
          this.selectedIllustration(this.model.illustrations);
          this.formGroup.patchValue(this.model);
	if(this.model.price == 0){
            this.freeOfCostModel = this.freeOfCost;
          }
	  this.moduleTitle = "Edit application: " + this.model.title;
          this.pageTitle.setTitle(this.moduleTitle);
          //clinet want to edit the applicaiton event it is issued or commited
          // if (this.model.readonly) {       
          //   this.disableAllInput();
          // }
        }, (error: ServerResponse) => {
          spinner.hide();
          this.router.navigate(['manage/isbn-applications']);
        });
    }

  }

  // protected selectedIllustration(illustrations){
  //  
  //  
  //  
  //  
    // this.selectedItems=[
      // { item_id: 1, item_text: 'Chart' },
      // { item_id: 2, item_text: 'Map' }
    // ];
  // }
  protected selectedIllustration(illustrations){

    let illustrationsArray: Array<{ item_id: number, item_text: string }> = []; 

    illustrations.forEach(element => {
      illustrationsArray.push({ item_id: element.illustration_number, item_text: element.name });
    });

    this.selectedItems = illustrationsArray;
  }

  protected disableAllInput () {
    const controls = this.formGroup.controls;
    for (const key in controls) {
      const element: HTMLInputElement = <HTMLInputElement> document.getElementById(key);
      if (element) {
        element.disabled = true;
      }
    }
  }

  onRouteParamInitialized() {
    super.onRouteParamInitialized();
    this.setModelIfEditPage();
  }

  onItemSelect(item: any) {
    console.log(item);
  }
  onSelectAll(items: any) {
    console.log(items);
  }

  ngOnInit(): void {
    super.ngOnInit();
    this.pageTitle.setTitle("Create new application");
    this.initForm();
    // this.initDropdownValue();
    this.initDropdownSetting();
    this.config.itemTemplate = this.publisherSelectItemTemplate;
    this.config.equalityCheckProperty = 'id';
    this.config.actionButtonEnabled = false;
    this.config.disableEmptyField = true;
    this.setUploadOptions('book_cover_url', 'Book cover');
    this.setUploadOptions('deposit_slip_url', 'Deposit slip');
    this.languageService.get().subscribe((collection: Collection<any>) => {
      this.languageOptions = collection;
    });

  }
   
  
  public initDropdownSetting(){
    this.dropdownSettings = {
      singleSelection: false,
      idField: 'item_id',
      textField: 'item_text',
      selectAllText: 'Select All',
      unSelectAllText: 'UnSelect All',
      itemsShowLimit: 3,
      allowSearchFilter: false
    };
  }

  public initDropdownValue(){
    this.illustrationTypeOptions = [
      {item_id: IsbnIllustrationTypeEnum.Chart, item_text: 'Chart'},
      {item_id: IsbnIllustrationTypeEnum.Map, item_text: 'Map'},
      {item_id: IsbnIllustrationTypeEnum.Tables, item_text: 'Tables'},
      {item_id: IsbnIllustrationTypeEnum.None, item_text: 'None'},
  
    ];
  }
  public addAuthor () {
    const formArray: FormArray = <FormArray>this.formGroup.controls['authors'];
    formArray.push(this.formBuilder.group({
      name: ['']
    }));
  }

  public removeAuthor (index: number, force?: boolean) {

    const formArray: FormArray = <FormArray>this.formGroup.controls['authors'];
    if (formArray.length < 2 && !force ) {
      return;
    }
    if (formArray.length > 0) {
      formArray.removeAt(index);
    }
  }



  protected initForm() {
    this.formGroup = this.formBuilder.group({
      application_id: [null,],
      requested_date: [null, Validators.required],
      isbn_type: [null, Validators.required],
      title: [null, Validators.required],
      edition: [null, Validators.required],
      subject: [null, Validators.required],
      accession_number: [null],
      language: [null, Validators.required],
      type: [null],
      corporate_author: [null, ],
      corporate_author_address: [null, ],
      affiliation_author_address: [null, ],
      year: [null, Validators.required],
      preliminary_page: [null, ],
      textual_page: [null, ],
      number_of_pages: [null, ],
      currency: [null, Validators.required],
      price: [null, Validators.required],
      book_number: [null, ],
      keywords: [null, ],
      illustration_type: [null, Validators.required],
      illustration_color: [null, Validators.required],
      photo_illustration: [null, Validators.required],
      book_cover_url: [null, Validators.required ],
      last_isbn_number: [null],
      deposit_slip_url: [null],
      call_number: [null],
      publisher_id: [null, Validators.required],
      publisher: [null, Validators.required],
      authors: this.formBuilder.array([])
    });
    this.formGroup.controls['publisher'].valueChanges.subscribe((value: BasePublisher) => {
      this.formGroup.controls['publisher_id'].setValue(value.id);
    });
    this.addAuthor();
  }

  public onFileUploadComplete (event: ServerResponse, formControlName: string) {
    this.formGroup.controls[formControlName].setValue(event.body.path);
  }

  public save(): Observable<any> {
	  const subject = new Subject();
	  this.formGroup.value.requested_date = moment(this.formGroup.value.requested_date).format('YYYY-MM-DD');
    const subscription = super.save().subscribe((response: any) => {
      subject.next(response);
      if (subscription) {
        subscription.unsubscribe();
      }
      this.router.navigate(['manage/isbn-applications']);
    }, (err: any) => {
      subject.error(err);
      if (subscription) {
        subscription.unsubscribe();
      }
    });

    return subject;
  }

  public onCommitClick() {
    this.appService.showConfirmDialog({
      confirmLabel: "Commit",
      message: "Are you sure you want to commit this change? Once committed you will not be able to edit this application.",
      onConfirm: () => {
        const spinner = this.ajaxSpinner.showSpinner();
        const subscription = (<IsbnApplicationService>(this.service)).commit(this.model).subscribe(() => {
          spinner.hide();
          this.model.readonly = 1;
          if (subscription) {
            subscription.unsubscribe();
          }
          this.showSuccessToast("The application '" + this.model.title +  "' has been committed as final.");
          this.router.navigate(['/manage/isbn-applications']);
        }, () => {
          spinner.hide();
          if (subscription) {
            subscription.unsubscribe();
          }
        });
      }
    });
  }

  public showIsbnModalBox (application: BaseIsbnApplication, isbn?: BaseIssuedISBN) {
    this.nicoCtrl.create(IsbnIssueModalComponent, {
      htmlTemplateAsIs: true,
      data: {application: application, isbn: isbn, isbnAddOn: null},
      onOkayProcessComplete: ( model: any, message: any) => {
        this.showSuccessToast(message);
        this.getList();
      }
    }).present();
  }

  public getBack()
  {
    Helper.setStorage('getFiltered',true);
    this.router.navigate(['manage/isbn-applications']);
    }

  public onChangePrice(event: any) {
    console.log(event.target.value);
    if(event.target.value == 0) {
      this.freeOfCostModel = this.freeOfCost;
    } else {
      this.freeOfCostModel = '';
    }
  }
}
