/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,missingOverride,missingReturn,unusedPrivateMembers,uselessCode} checked by tsc
 */
import { Component, Inject, Input, ViewChild, ContentChild, ElementRef, ChangeDetectorRef, ChangeDetectionStrategy, PLATFORM_ID } from '@angular/core';
import { isPlatformBrowser } from '@angular/common';
import { CdkScrollable } from '@angular/cdk/scrolling';
import { BreakpointObserver, Breakpoints } from '@angular/cdk/layout';
import { fromEvent, Subject } from 'rxjs';
import { takeUntil, tap, throttleTime } from 'rxjs/operators';
import { SmoothScroll } from '../smooth-scroll/smooth-scroll';
import { NgScrollbarView } from './ng-scrollbar-view';
// Native scrollbar size is 17px on all browsers,
// This value will be used to push the native scrollbar out of the scroll view to hide them
// An extra 1px is added to hide them properly on Edge browser
/** @type {?} */
const NATIVE_SCROLLBAR_SIZE = '18px';
export class NgScrollbar {
    /**
     * @param {?} _changeDetectorRef
     * @param {?} _breakpointObserver
     * @param {?} _platform
     */
    constructor(_changeDetectorRef, _breakpointObserver, _platform) {
        this._changeDetectorRef = _changeDetectorRef;
        this._breakpointObserver = _breakpointObserver;
        this._platform = _platform;
        /**
         * Horizontal custom scrollbar
         */
        this.trackX = false;
        /**
         * Vertical custom Scrollbar
         */
        this.trackY = true;
        /**
         * Scrollbar visibility
         */
        this.shown = 'native';
        /**
         * Auto update scrollbars on content changes (Mutation Observer)
         */
        this.autoUpdate = true;
        /**
         * The smooth scroll duration when a scrollbar is clicked
         */
        this.scrollToDuration = 300;
        /**
         * Disable custom scrollbars on specific breakpoints
         */
        this.disableOnBreakpoints = [
            Breakpoints.HandsetLandscape,
            Breakpoints.HandsetPortrait
        ];
        this._disabled = false;
        /**
         * Unsubscribe component observables on destroy
         */
        this._unsubscribe$ = new Subject();
        /**
         * Steam that emits when scrollbar thumbnail needs to update (for internal uses)
         */
        this._updateObserver = new Subject();
        this.updateObserver = this._updateObserver.asObservable();
    }
    /**
     * Disable custom scrollbars and switch back to native scrollbars
     * @return {?}
     */
    get disabled() {
        return this._disabled;
    }
    /**
     * @param {?} disable
     * @return {?}
     */
    set disabled(disable) {
        disable ? this.disable() : this.enable();
    }
    /**
     * Viewport Element
     * @return {?}
     */
    get view() {
        return this.customViewPort
            ? this.customViewPort.virtualScrollViewport.getElementRef().nativeElement
            : this.scrollViewport.getElementRef().nativeElement;
    }
    /**
     * @return {?}
     */
    get scrollable() {
        return this.customViewPort
            ? this.customViewPort.virtualScrollViewport
            : this.scrollViewport;
    }
    /**
     * @return {?}
     */
    get smoothScroll() {
        return this.customViewPort
            ? this.customViewPort.smoothScroll
            : this.viewSmoothScroll;
    }
    /**
     * @return {?}
     */
    get hideNativeScrollbars() {
        /** @type {?} */
        const size = this.disabled ? '100%' : `calc(100% + ${NATIVE_SCROLLBAR_SIZE})`;
        return {
            width: this.trackY ? size : '100%',
            height: this.trackX ? size : '100%'
        };
    }
    /**
     * @return {?}
     */
    showScrollbarY() {
        return this.shown === 'always' || this.view.scrollHeight > this.view.clientHeight;
    }
    /**
     * @return {?}
     */
    showScrollbarX() {
        return this.shown === 'always' || this.view.scrollWidth > this.view.clientWidth;
    }
    /**
     * @return {?}
     */
    ngAfterViewInit() {
        // Avoid 'expression has changed after it was checked' error when 'disableOnBreakpoints' is set to false
        Promise.resolve().then((/**
         * @return {?}
         */
        () => {
            if (!this.disabled) {
                if (this.disableOnBreakpoints) {
                    // Enable/Disable custom scrollbar on breakpoints (Used to disable scrollbars on mobile phones)
                    this._breakpointObserver.observe(this.disableOnBreakpoints).pipe(tap((/**
                     * @param {?} result
                     * @return {?}
                     */
                    (result) => result.matches ? this.disable() : this.enable())), takeUntil(this._unsubscribe$)).subscribe();
                }
                else {
                    this.enable();
                }
            }
            // Update state on content changes
            this.updateObserver.pipe(throttleTime(200), tap((/**
             * @return {?}
             */
            () => this._changeDetectorRef.markForCheck())), takeUntil(this._unsubscribe$)).subscribe();
            if (isPlatformBrowser(this._platform)) {
                // Update on window resize
                fromEvent(window, 'resize').pipe(throttleTime(200), tap((/**
                 * @return {?}
                 */
                () => this.update())), takeUntil(this._unsubscribe$)).subscribe();
            }
        }));
    }
    /**
     * @return {?}
     */
    ngOnDestroy() {
        this._unsubscribe$.next();
        this._unsubscribe$.complete();
        if (this._observer) {
            this._observer.disconnect();
        }
    }
    /**
     * Update scrollbar thumbnail position
     * @return {?}
     */
    update() {
        if (!this.disabled) {
            this._updateObserver.next();
        }
    }
    /**
     * Enable custom scrollbar
     * @return {?}
     */
    enable() {
        if (this.view) {
            this._disabled = false;
            // Update view
            this._changeDetectorRef.markForCheck();
            if (!this.customViewPort && this.autoUpdate && isPlatformBrowser(this._platform)) {
                // Observe content changes
                this._observer = new MutationObserver((/**
                 * @return {?}
                 */
                () => this.update()));
                this._observer.observe(this.view, { subtree: true, childList: true, characterData: true });
            }
        }
    }
    /**
     * Disable custom scrollbar
     * @return {?}
     */
    disable() {
        this._disabled = true;
        if (this._observer) {
            this._observer.disconnect();
        }
    }
    /**
     * @param {?} options
     * @return {?}
     */
    scrollTo(options) {
        return this.smoothScroll.scrollTo(options);
    }
    /**
     * @param {?} selector
     * @param {?=} offset
     * @param {?=} duration
     * @param {?=} easeFunc
     * @return {?}
     */
    scrollToElement(selector, offset = 0, duration, easeFunc) {
        return this.smoothScroll.scrollToElement(selector, offset, duration, easeFunc);
    }
    /**
     * @param {?} to
     * @param {?=} duration
     * @param {?=} easeFunc
     * @return {?}
     */
    scrollXTo(to, duration, easeFunc) {
        return this.smoothScroll.scrollXTo(to, duration, easeFunc);
    }
    /**
     * @param {?} to
     * @param {?=} duration
     * @param {?=} easeFunc
     * @return {?}
     */
    scrollYTo(to, duration, easeFunc) {
        return this.smoothScroll.scrollYTo(to, duration, easeFunc);
    }
    /**
     * @param {?=} duration
     * @param {?=} easeFunc
     * @return {?}
     */
    scrollToTop(duration, easeFunc) {
        return this.smoothScroll.scrollToTop(duration, easeFunc);
    }
    /**
     * @param {?=} duration
     * @param {?=} easeFunc
     * @return {?}
     */
    scrollToBottom(duration, easeFunc) {
        return this.smoothScroll.scrollToBottom(duration, easeFunc);
    }
    /**
     * @param {?=} duration
     * @param {?=} easeFunc
     * @return {?}
     */
    scrollToRight(duration, easeFunc) {
        return this.smoothScroll.scrollToRight(duration, easeFunc);
    }
    /**
     * @param {?=} duration
     * @param {?=} easeFunc
     * @return {?}
     */
    scrollToLeft(duration, easeFunc) {
        return this.smoothScroll.scrollToLeft(duration, easeFunc);
    }
}
NgScrollbar.decorators = [
    { type: Component, args: [{
                selector: 'ng-scrollbar',
                template: "<div class=\"ng-scrollbar-layout ng-scrollbar-x-layout\"\r\n     [class.ng-scrollbar-invert]=\"invertX\">\r\n\r\n  <div class=\"ng-scrollbar-layout ng-scrollbar-y-layout\"\r\n       [class.ng-scrollbar-invert]=\"invertY\">\r\n\r\n    <div class=\"ng-scroll-view-container\">\r\n      <div #view\r\n           cdkScrollable\r\n           smoothScroll\r\n           class=\"ng-scroll-view {{viewClass}}\"\r\n           [ngStyle]=\"hideNativeScrollbars\">\r\n        <ng-content></ng-content>\r\n      </div>\r\n    </div>\r\n\r\n    <ng-scrollbar-y #y\r\n                    *ngIf=\"!disabled && trackY\"\r\n                    [class.ng-scrollbar-visible]=\"showScrollbarY()\"\r\n                    [barClass]=\"barClass\"\r\n                    [thumbClass]=\"thumbClass\"\r\n                    [scrollToDuration]=\"scrollToDuration\">\r\n    </ng-scrollbar-y>\r\n  </div>\r\n\r\n  <ng-scrollbar-x #x\r\n                  *ngIf=\"!disabled && trackX\"\r\n                  [class.ng-scrollbar-visible]=\"showScrollbarX()\"\r\n                  [barClass]=\"barClass\"\r\n                  [thumbClass]=\"thumbClass\"\r\n                  [scrollToDuration]=\"scrollToDuration\">\r\n  </ng-scrollbar-x>\r\n\r\n</div>\r\n",
                changeDetection: ChangeDetectionStrategy.OnPush,
                host: {
                    '[attr.customView]': '!!customViewPort',
                    '[attr.trackX]': 'trackX',
                    '[attr.trackY]': 'trackY',
                    '[attr.compact]': 'compact',
                    '[attr.autoHide]': 'shown === "hover"'
                },
                styles: [":host{display:block;overflow:hidden;--scrollbar-color:transparent;--scrollbar-container-color:transparent;--scrollbar-thumb-color:rgba(0, 0, 0, 0.2);--scrollbar-thumb-hover-color:rgba(0, 0, 0, 0.3);--scrollbar-border-radius:4px;--scrollbar-size:6px;--scrollbar-padding:8px;--scroll-view-margin:0;--scroll-view-color:transparent}:host[trackY=true]>.ng-scrollbar-layout>.ng-scrollbar-layout>.ng-scroll-view-container>.ng-scroll-view{overflow-y:scroll}:host[trackX=true]>.ng-scrollbar-layout>.ng-scrollbar-layout>.ng-scroll-view-container>.ng-scroll-view{overflow-x:scroll}:host>.ng-scrollbar-x-layout{flex-direction:column}:host>.ng-scrollbar-x-layout.ng-scrollbar-invert{flex-direction:column-reverse}:host>.ng-scrollbar-x-layout>.ng-scrollbar-y-layout{flex-direction:row}:host>.ng-scrollbar-x-layout>.ng-scrollbar-y-layout.ng-scrollbar-invert{flex-direction:row-reverse}:host[compact=true]>.ng-scrollbar-x-layout>ng-scrollbar-x{position:absolute;bottom:0}:host[compact=true]>.ng-scrollbar-x-layout.ng-scrollbar-invert>ng-scrollbar-x{top:0;bottom:unset}:host[compact=true]>.ng-scrollbar-x-layout>.ng-scrollbar-y-layout>ng-scrollbar-y{position:absolute;right:0;left:unset}:host[compact=true]>.ng-scrollbar-x-layout>.ng-scrollbar-y-layout.ng-scrollbar-invert>ng-scrollbar-y{right:unset;left:0}:host[autoHide=true]>.ng-scrollbar-layout>.ng-scrollbar-layout>ng-scrollbar-y,:host[autoHide=true]>.ng-scrollbar-layout>ng-scrollbar-x{opacity:0;transition:opacity 120ms ease-out}:host[autoHide=true]:active>.ng-scrollbar-layout>.ng-scrollbar-layout>ng-scrollbar-y,:host[autoHide=true]:active>.ng-scrollbar-layout>ng-scrollbar-x,:host[autoHide=true]:focus>.ng-scrollbar-layout>.ng-scrollbar-layout>ng-scrollbar-y,:host[autoHide=true]:focus>.ng-scrollbar-layout>ng-scrollbar-x,:host[autoHide=true]:hover>.ng-scrollbar-layout>.ng-scrollbar-layout>ng-scrollbar-y,:host[autoHide=true]:hover>.ng-scrollbar-layout>ng-scrollbar-x{opacity:1;transition:opacity 340ms ease-out}.ng-scroll-view,.ng-scrollbar-layout,:host{position:relative;height:100%;width:100%}.ng-scrollbar-layout{display:flex;min-height:0}.ng-scroll-view-container{flex:1;position:relative;overflow:hidden;margin:var(--scroll-view-margin)}.ng-scroll-view{box-sizing:content-box;-webkit-transform:translateZ(0);transform:translateZ(0);background:var(--scroll-view-color);-webkit-overflow-scrolling:touch}ng-scrollbar-x,ng-scrollbar-y{display:none;box-sizing:border-box;padding:var(--scrollbar-padding);background:var(--scrollbar-container-color)}ng-scrollbar-x.ng-scrollbar-visible,ng-scrollbar-y.ng-scrollbar-visible{display:block}ng-scrollbar-y{top:0;bottom:0}ng-scrollbar-x{left:0;right:0}::ng-deep ng-scrollbar-y .ng-scrollbar{width:var(--scrollbar-size)}::ng-deep ng-scrollbar-y .ng-scrollbar-thumb{width:100%}::ng-deep ng-scrollbar-x .ng-scrollbar{height:var(--scrollbar-size)}::ng-deep ng-scrollbar-x .ng-scrollbar-thumb{height:100%}::ng-deep .ng-scrollbar{height:100%;width:100%;z-index:1;border-radius:var(--scrollbar-border-radius);background-color:var(--scrollbar-color)}::ng-deep .ng-scrollbar-thumb{box-sizing:border-box;position:relative;width:0;height:0;border-radius:inherit;background-color:var(--scrollbar-thumb-color);-webkit-transform:translate3d(0,0,0);transform:translate3d(0,0,0);transition:height 150ms ease-out}::ng-deep .ng-scrollbar-thumb:active,::ng-deep .ng-scrollbar-thumb:hover{background-color:var(--scrollbar-thumb-hover-color)}::ng-deep cdk-virtual-scroll-viewport{height:100%}::ng-deep ng-scrollbar[customView=true] .ng-scroll-view{overflow:hidden!important}::ng-deep ng-scrollbar[trackY=true]>.ng-scrollbar-layout>.ng-scrollbar-layout>.ng-scroll-view-container>.ng-scroll-view,::ng-deep ng-scrollbar[trackY=true]>.ng-scrollbar-layout>.ng-scrollbar-layout>.ng-scroll-view-container>cdk-virtual-scroll-viewport{overflow-y:scroll}::ng-deep ng-scrollbar[trackX=true]>.ng-scrollbar-layout>.ng-scrollbar-layout>.ng-scroll-view-container>.ng-scroll-view,::ng-deep ng-scrollbar[trackX=true]>.ng-scrollbar-layout>.ng-scrollbar-layout>.ng-scroll-view-container>cdk-virtual-scroll-viewport{overflow-x:scroll}"]
            }] }
];
/** @nocollapse */
NgScrollbar.ctorParameters = () => [
    { type: ChangeDetectorRef },
    { type: BreakpointObserver },
    { type: Object, decorators: [{ type: Inject, args: [PLATFORM_ID,] }] }
];
NgScrollbar.propDecorators = {
    trackX: [{ type: Input }],
    trackY: [{ type: Input }],
    shown: [{ type: Input }],
    autoUpdate: [{ type: Input }],
    viewClass: [{ type: Input }],
    barClass: [{ type: Input }],
    thumbClass: [{ type: Input }],
    scrollToDuration: [{ type: Input }],
    compact: [{ type: Input }],
    invertY: [{ type: Input }],
    invertX: [{ type: Input }],
    disableOnBreakpoints: [{ type: Input }],
    disabled: [{ type: Input, args: ['disabled',] }],
    verticalScrollbar: [{ type: ViewChild, args: ['y', { read: ElementRef },] }],
    horizontalScrollbar: [{ type: ViewChild, args: ['x', { read: ElementRef },] }],
    scrollViewport: [{ type: ViewChild, args: [CdkScrollable,] }],
    viewSmoothScroll: [{ type: ViewChild, args: [SmoothScroll,] }],
    customViewPort: [{ type: ContentChild, args: [NgScrollbarView,] }]
};
if (false) {
    /**
     * Horizontal custom scrollbar
     * @type {?}
     */
    NgScrollbar.prototype.trackX;
    /**
     * Vertical custom Scrollbar
     * @type {?}
     */
    NgScrollbar.prototype.trackY;
    /**
     * Scrollbar visibility
     * @type {?}
     */
    NgScrollbar.prototype.shown;
    /**
     * Auto update scrollbars on content changes (Mutation Observer)
     * @type {?}
     */
    NgScrollbar.prototype.autoUpdate;
    /**
     * Viewport class
     * @type {?}
     */
    NgScrollbar.prototype.viewClass;
    /**
     * Scrollbars class
     * @type {?}
     */
    NgScrollbar.prototype.barClass;
    /**
     * Scrollbars thumbnails class
     * @type {?}
     */
    NgScrollbar.prototype.thumbClass;
    /**
     * The smooth scroll duration when a scrollbar is clicked
     * @type {?}
     */
    NgScrollbar.prototype.scrollToDuration;
    /**
     * Compact mode
     * @type {?}
     */
    NgScrollbar.prototype.compact;
    /**
     * Invert vertical scrollbar position, if set the scrollbar will be on the right
     * @type {?}
     */
    NgScrollbar.prototype.invertY;
    /**
     * Invert horizontal scrollbar position, if set the scrollbar will go the top
     * @type {?}
     */
    NgScrollbar.prototype.invertX;
    /**
     * Disable custom scrollbars on specific breakpoints
     * @type {?}
     */
    NgScrollbar.prototype.disableOnBreakpoints;
    /**
     * @type {?}
     * @private
     */
    NgScrollbar.prototype._disabled;
    /**
     * Scrollbars ElementRef
     * @type {?}
     */
    NgScrollbar.prototype.verticalScrollbar;
    /** @type {?} */
    NgScrollbar.prototype.horizontalScrollbar;
    /**
     * Default viewport and smoothScroll references
     * @type {?}
     */
    NgScrollbar.prototype.scrollViewport;
    /** @type {?} */
    NgScrollbar.prototype.viewSmoothScroll;
    /**
     * Virtual viewport and smoothScroll references
     * @type {?}
     */
    NgScrollbar.prototype.customViewPort;
    /**
     * Unsubscribe component observables on destroy
     * @type {?}
     * @private
     */
    NgScrollbar.prototype._unsubscribe$;
    /**
     * Observe content changes
     * @type {?}
     * @private
     */
    NgScrollbar.prototype._observer;
    /**
     * Steam that emits when scrollbar thumbnail needs to update (for internal uses)
     * @type {?}
     * @private
     */
    NgScrollbar.prototype._updateObserver;
    /** @type {?} */
    NgScrollbar.prototype.updateObserver;
    /**
     * @type {?}
     * @private
     */
    NgScrollbar.prototype._changeDetectorRef;
    /**
     * @type {?}
     * @private
     */
    NgScrollbar.prototype._breakpointObserver;
    /**
     * @type {?}
     * @private
     */
    NgScrollbar.prototype._platform;
}
//# sourceMappingURL=data:application/json;base64,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