import { Component, Input, Output, EventEmitter, NgZone, Renderer2, ViewChild, NgModule } from '@angular/core';
import { timer } from 'rxjs';
import { CommonModule } from '@angular/common';

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,uselessCode} checked by tsc
 */
/**
 * fileEntry is an instance of {\@link FileSystemFileEntry} or {\@link FileSystemDirectoryEntry}.
 * Which one is it can be checked using {\@link FileSystemEntry.isFile} or {\@link FileSystemEntry.isDirectory}
 * properties of the given {\@link FileSystemEntry}.
 */
class UploadFile {
    /**
     * @param {?} relativePath
     * @param {?} fileEntry
     */
    constructor(relativePath, fileEntry) {
        this.relativePath = relativePath;
        this.fileEntry = fileEntry;
    }
}

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,uselessCode} checked by tsc
 */
class UploadEvent {
    /**
     * @param {?} files
     */
    constructor(files) {
        this.files = files;
    }
}

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,uselessCode} checked by tsc
 */
class FileComponent {
    /**
     * @param {?} zone
     * @param {?} renderer
     */
    constructor(zone, renderer) {
        this.zone = zone;
        this.renderer = renderer;
        this.accept = '*';
        this.dropZoneLabel = '';
        this.dropZoneClassName = 'ngx-file-drop__drop-zone drop-zone';
        this.contentClassName = 'ngx-file-drop__content content';
        this.showBrowseBtn = false;
        this.browseBtnClassName = 'btn btn-primary btn-xs ngx-file-drop__browse-btn';
        this.browseBtnLabel = 'Browse files';
        this.onFileDrop = new EventEmitter();
        this.onFileOver = new EventEmitter();
        this.onFileLeave = new EventEmitter();
        this.isDraggingOverDropZone = false;
        this.globalDraggingInProgress = false;
        this.files = [];
        this.numOfActiveReadEntries = 0;
        this.helperFormEl = null;
        this.fileInputPlaceholderEl = null;
        this.dropEventTimerSubscription = null;
        this._disabled = false;
        this.globalDragStartListener = this.renderer.listen('document', 'dragstart', (evt) => {
            this.globalDraggingInProgress = true;
        });
        this.globalDragEndListener = this.renderer.listen('document', 'dragend', (evt) => {
            this.globalDraggingInProgress = false;
        });
    }
    /**
     * @deprecated Will be removed in the next major version. Use `dropZoneLabel` instead.
     * @return {?}
     */
    get headertext() { return this.dropZoneLabel; }
    /**
     * @deprecated Will be removed in the next major version. Use `dropZoneLabel` instead.
     * @param {?} value
     * @return {?}
     */
    set headertext(value) {
        this.dropZoneLabel = value;
    }
    /**
     * @deprecated Will be removed in the next major version. Use `dropZoneClassName` instead.
     * @return {?}
     */
    get customstyle() { return this.dropZoneClassName; }
    /**
     * @deprecated Will be removed in the next major version. Use `dropZoneClassName` instead.
     * @param {?} value
     * @return {?}
     */
    set customstyle(value) {
        this.dropZoneClassName = value;
    }
    /**
     * @deprecated Will be removed in the next major version. Use `contentClassName` instead.
     * @return {?}
     */
    get customContentStyle() { return this.contentClassName; }
    /**
     * @deprecated Will be removed in the next major version. Use `contentClassName` instead.
     * @param {?} value
     * @return {?}
     */
    set customContentStyle(value) {
        this.contentClassName = value;
    }
    /**
     * @return {?}
     */
    get disabled() { return this._disabled; }
    /**
     * @param {?} value
     * @return {?}
     */
    set disabled(value) {
        this._disabled = (value != null && `${value}` !== 'false');
    }
    /**
     * @deprecated Will be removed in the next major version. Use `disabled` instead.
     * @return {?}
     */
    get disableIf() { return this.disabled; }
    /**
     * @deprecated Will be removed in the next major version. Use `disabled` instead.
     * @param {?} value
     * @return {?}
     */
    set disableIf(value) {
        this.disabled = value;
    }
    /**
     * @deprecated Will be removed in the next major version. Use `browseBtnClassName` instead.
     * @return {?}
     */
    get customBtnStyling() { return this.browseBtnClassName; }
    /**
     * @deprecated Will be removed in the next major version. Use `browseBtnClassName` instead.
     * @param {?} value
     * @return {?}
     */
    set customBtnStyling(value) {
        this.browseBtnClassName = value;
    }
    /**
     * @return {?}
     */
    ngOnDestroy() {
        if (this.dropEventTimerSubscription) {
            this.dropEventTimerSubscription.unsubscribe();
            this.dropEventTimerSubscription = null;
        }
        this.globalDragStartListener();
        this.globalDragEndListener();
        this.files = [];
        this.helperFormEl = null;
        this.fileInputPlaceholderEl = null;
    }
    /**
     * @param {?} event
     * @return {?}
     */
    onDragOver(event) {
        if (!this.isDropzoneDisabled()) {
            if (!this.isDraggingOverDropZone) {
                this.isDraggingOverDropZone = true;
                this.onFileOver.emit(event);
            }
            this.preventAndStop(event);
        }
    }
    /**
     * @param {?} event
     * @return {?}
     */
    onDragLeave(event) {
        if (!this.isDropzoneDisabled()) {
            if (this.isDraggingOverDropZone) {
                this.isDraggingOverDropZone = false;
                this.onFileLeave.emit(event);
            }
            this.preventAndStop(event);
        }
    }
    /**
     * @param {?} event
     * @return {?}
     */
    dropFiles(event) {
        if (!this.isDropzoneDisabled()) {
            this.isDraggingOverDropZone = false;
            if (event.dataTransfer) {
                event.dataTransfer.dropEffect = 'copy';
                /** @type {?} */
                let items;
                if (event.dataTransfer.items) {
                    items = event.dataTransfer.items;
                }
                else {
                    items = event.dataTransfer.files;
                }
                this.preventAndStop(event);
                this.checkFiles(items);
            }
        }
    }
    /**
     * @param {?} event
     * @return {?}
     */
    onBrowseButtonClick(event) {
        if (this.fileSelector && this.fileSelector.nativeElement) {
            ((/** @type {?} */ (this.fileSelector.nativeElement))).click();
        }
    }
    /**
     * Processes the change event of the file input and adds the given files.
     * @param {?} event
     * @return {?}
     */
    uploadFiles(event) {
        if (!this.isDropzoneDisabled()) {
            if (event.target) {
                /** @type {?} */
                const items = ((/** @type {?} */ (event.target))).files || ((/** @type {?} */ ([])));
                this.checkFiles(items);
                this.resetFileInput();
            }
        }
    }
    /**
     * @param {?} items
     * @return {?}
     */
    checkFiles(items) {
        for (let i = 0; i < items.length; i++) {
            /** @type {?} */
            const item = items[i];
            /** @type {?} */
            let entry = null;
            if (this.canGetAsEntry(item)) {
                entry = item.webkitGetAsEntry();
            }
            if (!entry) {
                if (item) {
                    /** @type {?} */
                    const fakeFileEntry = {
                        name: ((/** @type {?} */ (item))).name,
                        isDirectory: false,
                        isFile: true,
                        file: (callback) => {
                            callback((/** @type {?} */ (item)));
                        },
                    };
                    /** @type {?} */
                    const toUpload = new UploadFile(fakeFileEntry.name, fakeFileEntry);
                    this.addToQueue(toUpload);
                }
            }
            else {
                if (entry.isFile) {
                    /** @type {?} */
                    const toUpload = new UploadFile(entry.name, entry);
                    this.addToQueue(toUpload);
                }
                else if (entry.isDirectory) {
                    this.traverseFileTree(entry, entry.name);
                }
            }
        }
        if (this.dropEventTimerSubscription) {
            this.dropEventTimerSubscription.unsubscribe();
        }
        this.dropEventTimerSubscription = timer(200, 200)
            .subscribe(() => {
            if (this.files.length > 0 && this.numOfActiveReadEntries === 0) {
                this.onFileDrop.emit(new UploadEvent(this.files));
                this.files = [];
            }
        });
    }
    /**
     * @param {?} item
     * @param {?} path
     * @return {?}
     */
    traverseFileTree(item, path) {
        if (item.isFile) {
            /** @type {?} */
            const toUpload = new UploadFile(path, item);
            this.files.push(toUpload);
        }
        else {
            path = path + '/';
            /** @type {?} */
            const dirReader = ((/** @type {?} */ (item))).createReader();
            /** @type {?} */
            let entries = [];
            /** @type {?} */
            const readEntries = () => {
                this.numOfActiveReadEntries++;
                dirReader.readEntries((result) => {
                    if (!result.length) {
                        // add empty folders
                        if (entries.length === 0) {
                            /** @type {?} */
                            const toUpload = new UploadFile(path, item);
                            this.zone.run(() => {
                                this.addToQueue(toUpload);
                            });
                        }
                        else {
                            for (let i = 0; i < entries.length; i++) {
                                this.zone.run(() => {
                                    this.traverseFileTree(entries[i], path + entries[i].name);
                                });
                            }
                        }
                    }
                    else {
                        // continue with the reading
                        entries = entries.concat(result);
                        readEntries();
                    }
                    this.numOfActiveReadEntries--;
                });
            };
            readEntries();
        }
    }
    /**
     * Clears any added files from the file input element so the same file can subsequently be added multiple times.
     * @return {?}
     */
    resetFileInput() {
        if (this.fileSelector && this.fileSelector.nativeElement) {
            /** @type {?} */
            const fileInputEl = (/** @type {?} */ (this.fileSelector.nativeElement));
            /** @type {?} */
            const fileInputContainerEl = fileInputEl.parentElement;
            /** @type {?} */
            const helperFormEl = this.getHelperFormElement();
            /** @type {?} */
            const fileInputPlaceholderEl = this.getFileInputPlaceholderElement();
            // Just a quick check so we do not mess up the DOM (will never happen though).
            if (fileInputContainerEl !== helperFormEl) {
                // Insert the form input placeholder in the DOM before the form input element.
                this.renderer.insertBefore(fileInputContainerEl, fileInputPlaceholderEl, fileInputEl);
                // Add the form input as child of the temporary form element, removing the form input from the DOM.
                this.renderer.appendChild(helperFormEl, fileInputEl);
                // Reset the form, thus clearing the input element of any files.
                helperFormEl.reset();
                // Add the file input back to the DOM in place of the file input placeholder element.
                this.renderer.insertBefore(fileInputContainerEl, fileInputEl, fileInputPlaceholderEl);
                // Remove the input placeholder from the DOM
                this.renderer.removeChild(fileInputContainerEl, fileInputPlaceholderEl);
            }
        }
    }
    /**
     * Get a cached HTML form element as a helper element to clear the file input element.
     * @return {?}
     */
    getHelperFormElement() {
        if (!this.helperFormEl) {
            this.helperFormEl = (/** @type {?} */ (this.renderer.createElement('form')));
        }
        return this.helperFormEl;
    }
    /**
     * Get a cached HTML div element to be used as placeholder for the file input element when clearing said element.
     * @return {?}
     */
    getFileInputPlaceholderElement() {
        if (!this.fileInputPlaceholderEl) {
            this.fileInputPlaceholderEl = (/** @type {?} */ (this.renderer.createElement('div')));
        }
        return this.fileInputPlaceholderEl;
    }
    /**
     * @param {?} item
     * @return {?}
     */
    canGetAsEntry(item) {
        return !!item.webkitGetAsEntry;
    }
    /**
     * @return {?}
     */
    isDropzoneDisabled() {
        return (this.globalDraggingInProgress || this.disabled);
    }
    /**
     * @param {?} item
     * @return {?}
     */
    addToQueue(item) {
        this.files.push(item);
    }
    /**
     * @param {?} event
     * @return {?}
     */
    preventAndStop(event) {
        event.stopPropagation();
        event.preventDefault();
    }
}
FileComponent.decorators = [
    { type: Component, args: [{
                selector: 'file-drop',
                template: "<div [className]=\"dropZoneClassName\"\r\n     [class.over]=\"isDraggingOverDropZone\"\r\n     [class.ngx-file-drop__drop-zone--over]=\"isDraggingOverDropZone\"\r\n     (drop)=\"dropFiles($event)\"\r\n     (dragover)=\"onDragOver($event)\"\r\n     (dragleave)=\"onDragLeave($event)\">\r\n    <div [className]=\"contentClassName\">\r\n        <ng-content></ng-content>\r\n        <div *ngIf=\"dropZoneLabel\" class=\"ngx-file-drop__drop-zone-label\">{{dropZoneLabel}}</div>\r\n        <div *ngIf=\"showBrowseBtn\">\r\n            <input type=\"file\" #fileSelector [accept]=\"accept\" (change)=\"uploadFiles($event)\" multiple class=\"ngx-file-drop__file-input\" />\r\n            <input type=\"button\" [className]=\"browseBtnClassName\" value=\"{{browseBtnLabel}}\" (click)=\"onBrowseButtonClick($event)\" />\r\n        </div>\r\n    </div>\r\n</div>\r\n",
                styles: [".drop-zone{height:100px;margin:auto;border:2px dotted #0782d0;border-radius:30px}.over{background-color:rgba(147,147,147,.5)}.content{display:flex;align-items:center;justify-content:center;height:100px;color:#0782d0}.ngx-file-drop__drop-zone-label{text-align:center}.ngx-file-drop__file-input{display:none}"]
            }] }
];
/** @nocollapse */
FileComponent.ctorParameters = () => [
    { type: NgZone },
    { type: Renderer2 }
];
FileComponent.propDecorators = {
    accept: [{ type: Input }],
    dropZoneLabel: [{ type: Input }],
    headertext: [{ type: Input }],
    dropZoneClassName: [{ type: Input }],
    customstyle: [{ type: Input }],
    contentClassName: [{ type: Input }],
    customContentStyle: [{ type: Input }],
    disabled: [{ type: Input }],
    disableIf: [{ type: Input }],
    showBrowseBtn: [{ type: Input }],
    browseBtnClassName: [{ type: Input }],
    customBtnStyling: [{ type: Input }],
    browseBtnLabel: [{ type: Input }],
    onFileDrop: [{ type: Output }],
    onFileOver: [{ type: Output }],
    onFileLeave: [{ type: Output }],
    fileSelector: [{ type: ViewChild, args: ['fileSelector',] }]
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,uselessCode} checked by tsc
 */
class FileDropModule {
}
FileDropModule.decorators = [
    { type: NgModule, args: [{
                declarations: [
                    FileComponent,
                ],
                exports: [FileComponent],
                imports: [CommonModule],
                providers: [],
                bootstrap: [FileComponent],
            },] }
];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,uselessCode} checked by tsc
 */

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,uselessCode} checked by tsc
 */

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,uselessCode} checked by tsc
 */

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,uselessCode} checked by tsc
 */

export { FileComponent, FileDropModule, UploadFile, UploadEvent };

//# sourceMappingURL=data:application/json;charset=utf-8;base64,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