import * as tslib_1 from "tslib";
import { isPlatformBrowser } from '@angular/common';
import { Inject, Injectable, InjectionToken, NgZone, Optional, PLATFORM_ID } from '@angular/core';
import { Subject } from 'rxjs';
import { loadScript, RECAPTCHA_BASE_URL, RECAPTCHA_NONCE } from './recaptcha-loader.service';
export const RECAPTCHA_V3_SITE_KEY = new InjectionToken('recaptcha-v3-site-key');
/**
 * The main service for working with reCAPTCHA v3 APIs.
 *
 * Use the `execute` method for executing a single action, and
 * `onExecute` observable for listening to all actions at once.
 */
let ReCaptchaV3Service = class ReCaptchaV3Service {
    constructor(zone, siteKey, 
    // tslint:disable-next-line:no-any
    platformId, baseUrl, nonce) {
        /** @internal */
        this.onLoadComplete = (grecaptcha) => {
            this.grecaptcha = grecaptcha;
            if (this.actionBacklog && this.actionBacklog.length > 0) {
                this.actionBacklog.forEach(([action, subject]) => this.executeActionWithSubject(action, subject));
                this.actionBacklog = undefined;
            }
        };
        this.zone = zone;
        this.isBrowser = isPlatformBrowser(platformId);
        this.siteKey = siteKey;
        this.nonce = nonce;
        this.baseUrl = baseUrl;
        this.init();
    }
    get onExecute() {
        if (!this.onExecuteSubject) {
            this.onExecuteSubject = new Subject();
            this.onExecuteObservable = this.onExecuteSubject.asObservable();
        }
        return this.onExecuteObservable;
    }
    /**
     * Executes the provided `action` with reCAPTCHA v3 API.
     * Use the emitted token value for verification purposes on the backend.
     *
     * For more information about reCAPTCHA v3 actions and tokens refer to the official documentation at
     * https://developers.google.com/recaptcha/docs/v3.
     *
     * @param {string} action the action to execute
     * @returns {Observable<string>} an `Observable` that will emit the reCAPTCHA v3 string `token` value whenever ready.
     * The returned `Observable` completes immediately after emitting a value.
     */
    execute(action) {
        const subject = new Subject();
        if (this.isBrowser) {
            if (!this.grecaptcha) {
                // todo: add to array of later executions
                if (!this.actionBacklog) {
                    this.actionBacklog = [];
                }
                this.actionBacklog.push([action, subject]);
            }
            else {
                this.executeActionWithSubject(action, subject);
            }
        }
        return subject.asObservable();
    }
    /** @internal */
    executeActionWithSubject(action, subject) {
        this.zone.runOutsideAngular(() => {
            // tslint:disable-next-line:no-any
            this.grecaptcha.execute(this.siteKey, { action }).then((token) => {
                this.zone.run(() => {
                    subject.next(token);
                    subject.complete();
                    if (this.onExecuteSubject) {
                        this.onExecuteSubject.next({ action, token });
                    }
                });
            });
        });
    }
    /** @internal */
    init() {
        if (this.isBrowser) {
            if ('grecaptcha' in window) {
                this.grecaptcha = grecaptcha;
            }
            else {
                loadScript(this.siteKey, this.onLoadComplete, '', this.baseUrl, this.nonce);
            }
        }
    }
};
ReCaptchaV3Service = tslib_1.__decorate([
    Injectable(),
    tslib_1.__param(1, Inject(RECAPTCHA_V3_SITE_KEY)),
    tslib_1.__param(2, Inject(PLATFORM_ID)),
    tslib_1.__param(3, Optional()), tslib_1.__param(3, Inject(RECAPTCHA_BASE_URL)),
    tslib_1.__param(4, Optional()), tslib_1.__param(4, Inject(RECAPTCHA_NONCE)),
    tslib_1.__metadata("design:paramtypes", [NgZone, String, Object, String, String])
], ReCaptchaV3Service);
export { ReCaptchaV3Service };
//# sourceMappingURL=data:application/json;base64,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