"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
function stringToFileBuffer(str) {
    // If we're in Node...
    if (typeof Buffer !== 'undefined' && typeof Buffer.from === 'function') {
        const buf = Buffer.from(str);
        const ab = new ArrayBuffer(buf.length);
        const view = new Uint8Array(ab);
        for (let i = 0; i < buf.length; ++i) {
            view[i] = buf[i];
        }
        return ab;
    }
    else if (typeof TextEncoder !== 'undefined') {
        // Modern browsers implement TextEncode.
        return new TextEncoder('utf-8').encode(str).buffer;
    }
    else {
        // Slowest method but sure to be compatible with every platform.
        const buf = new ArrayBuffer(str.length * 2); // 2 bytes for each char
        const bufView = new Uint16Array(buf);
        for (let i = 0, strLen = str.length; i < strLen; i++) {
            bufView[i] = str.charCodeAt(i);
        }
        return buf;
    }
}
exports.stringToFileBuffer = stringToFileBuffer;
exports.fileBuffer = (strings, ...values) => {
    return stringToFileBuffer(String.raw(strings, ...values));
};
function fileBufferToString(fileBuffer) {
    if (fileBuffer.toString.length == 1) {
        return fileBuffer.toString('utf-8');
    }
    else if (typeof Buffer !== 'undefined') {
        return Buffer.from(fileBuffer).toString('utf-8');
    }
    else if (typeof TextDecoder !== 'undefined') {
        // Modern browsers implement TextEncode.
        return new TextDecoder('utf-8').decode(new Uint8Array(fileBuffer));
    }
    else {
        // Slowest method but sure to be compatible with every platform.
        const bufView = new Uint8Array(fileBuffer);
        const bufLength = bufView.length;
        let result = '';
        let chunkLength = Math.pow(2, 16) - 1;
        // We have to chunk it because String.fromCharCode.apply will throw
        // `Maximum call stack size exceeded` on big inputs.
        for (let i = 0; i < bufLength; i += chunkLength) {
            if (i + chunkLength > bufLength) {
                chunkLength = bufLength - i;
            }
            result += String.fromCharCode.apply(null, bufView.subarray(i, i + chunkLength));
        }
        return result;
    }
}
exports.fileBufferToString = fileBufferToString;
//# sourceMappingURL=data:application/json;base64,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