"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ts = require("typescript");
const benchmark_1 = require("./benchmark");
class CancellationToken {
    constructor() {
        this._isCancelled = false;
    }
    requestCancellation() {
        this._isCancelled = true;
    }
    isCancellationRequested() {
        return this._isCancelled;
    }
    throwIfCancellationRequested() {
        if (this.isCancellationRequested()) {
            throw new ts.OperationCanceledException();
        }
    }
}
exports.CancellationToken = CancellationToken;
function hasErrors(diags) {
    return diags.some(d => d.category === ts.DiagnosticCategory.Error);
}
exports.hasErrors = hasErrors;
function gatherDiagnostics(program, jitMode, benchmarkLabel, cancellationToken) {
    const allDiagnostics = [];
    let checkOtherDiagnostics = true;
    function checkDiagnostics(fn) {
        if (checkOtherDiagnostics) {
            const diags = fn(undefined, cancellationToken);
            if (diags) {
                allDiagnostics.push(...diags);
                checkOtherDiagnostics = !hasErrors(diags);
            }
        }
    }
    if (jitMode) {
        const tsProgram = program;
        // Check syntactic diagnostics.
        benchmark_1.time(`${benchmarkLabel}.gatherDiagnostics.ts.getSyntacticDiagnostics`);
        checkDiagnostics(tsProgram.getSyntacticDiagnostics.bind(tsProgram));
        benchmark_1.timeEnd(`${benchmarkLabel}.gatherDiagnostics.ts.getSyntacticDiagnostics`);
        // Check semantic diagnostics.
        benchmark_1.time(`${benchmarkLabel}.gatherDiagnostics.ts.getSemanticDiagnostics`);
        checkDiagnostics(tsProgram.getSemanticDiagnostics.bind(tsProgram));
        benchmark_1.timeEnd(`${benchmarkLabel}.gatherDiagnostics.ts.getSemanticDiagnostics`);
    }
    else {
        const angularProgram = program;
        // Check TypeScript syntactic diagnostics.
        benchmark_1.time(`${benchmarkLabel}.gatherDiagnostics.ng.getTsSyntacticDiagnostics`);
        checkDiagnostics(angularProgram.getTsSyntacticDiagnostics.bind(angularProgram));
        benchmark_1.timeEnd(`${benchmarkLabel}.gatherDiagnostics.ng.getTsSyntacticDiagnostics`);
        // Check TypeScript semantic and Angular structure diagnostics.
        benchmark_1.time(`${benchmarkLabel}.gatherDiagnostics.ng.getTsSemanticDiagnostics`);
        checkDiagnostics(angularProgram.getTsSemanticDiagnostics.bind(angularProgram));
        benchmark_1.timeEnd(`${benchmarkLabel}.gatherDiagnostics.ng.getTsSemanticDiagnostics`);
        // Check Angular semantic diagnostics
        benchmark_1.time(`${benchmarkLabel}.gatherDiagnostics.ng.getNgSemanticDiagnostics`);
        checkDiagnostics(angularProgram.getNgSemanticDiagnostics.bind(angularProgram));
        benchmark_1.timeEnd(`${benchmarkLabel}.gatherDiagnostics.ng.getNgSemanticDiagnostics`);
    }
    return allDiagnostics;
}
exports.gatherDiagnostics = gatherDiagnostics;
//# sourceMappingURL=data:application/json;base64,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