/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as tslib_1 from "tslib";
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { R3_COMPILE_COMPONENT, R3_COMPILE_DIRECTIVE, R3_COMPILE_PIPE } from '../ivy_switch';
import { makeDecorator, makePropDecorator } from '../util/decorators';
/**
 * Type of the Directive metadata.
 */
export var Directive = makeDecorator('Directive', function (dir) {
    if (dir === void 0) { dir = {}; }
    return dir;
}, undefined, undefined, function (type, meta) { return (R3_COMPILE_DIRECTIVE || (function () { }))(type, meta); });
/**
 * Component decorator and metadata.
 *
 * @usageNotes
 *
 * ### Using animations
 *
 * The following snippet shows an animation trigger in a component's
 * metadata. The trigger is attached to an element in the component's
 * template, using "@_trigger_name_", and a state expression that is evaluated
 * at run time to determine whether the animation should start.
 *
 * ```typescript
 * @Component({
 *   selector: 'animation-cmp',
 *   templateUrl: 'animation-cmp.html',
 *   animations: [
 *     trigger('myTriggerName', [
 *       state('on', style({ opacity: 1 }),
 *       state('off', style({ opacity: 0 }),
 *       transition('on => off', [
 *         animate("1s")
 *       ])
 *     ])
 *   ]
 * })
 * ```
 *
 * ```html
 * <!-- animation-cmp.html -->
 * <div @myTriggerName="expression">...</div>
 * ```
 *
 * ### Preserving whitespace
 *
 * Removing whitespace can greatly reduce AOT-generated code size, and speed up view creation.
 * As of Angular 6, default for `preserveWhitespaces` is false (whitespace is removed).
 * To change the default setting for all components in your application, set
 * the `preserveWhitespaces` option of the AOT compiler.
 *
 * Current implementation removes whitespace characters as follows:
 * - Trims all whitespaces at the beginning and the end of a template.
 * - Removes whitespace-only text nodes. For example,
 * `<button>Action 1</button>  <button>Action 2</button>` becomes
 * `<button>Action 1</button><button>Action 2</button>`.
 * - Replaces a series of whitespace characters in text nodes with a single space.
 * For example, `<span>\n some text\n</span>` becomes `<span> some text </span>`.
 * - Does NOT alter text nodes inside HTML tags such as `<pre>` or `<textarea>`,
 * where whitespace characters are significant.
 *
 * Note that these transformations can influence DOM nodes layout, although impact
 * should be minimal.
 *
 * You can override the default behavior to preserve whitespace characters
 * in certain fragments of a template. For example, you can exclude an entire
 * DOM sub-tree by using the `ngPreserveWhitespaces` attribute:
 *
 * ```html
 * <div ngPreserveWhitespaces>
 *     whitespaces are preserved here
 *     <span>    and here </span>
 * </div>
 * ```
 *
 * You can force a single space to be preserved in a text node by using `&ngsp;`,
 * which is replaced with a space character by Angular's template
 * compiler:
 *
 * ```html
 * <a>Spaces</a>&ngsp;<a>between</a>&ngsp;<a>links.</a>
 * <!-->compiled to be equivalent to:</>
 *  <a>Spaces</a> <a>between</a> <a>links.</a>
 * ```
 *
 * Note that sequences of `&ngsp;` are still collapsed to just one space character when
 * the `preserveWhitespaces` option is set to `false`.
 *
 * ```html
 * <a>before</a>&ngsp;&ngsp;&ngsp;<a>after</a>
 * <!-->compiled to be equivalent to:</>
 *  <a>Spaces</a> <a>between</a> <a>links.</a>
 * ```
 *
 * To preserve sequences of whitespace characters, use the
 * `ngPreserveWhitespaces` attribute.
 *
 * @Annotation
 */
export var Component = makeDecorator('Component', function (c) {
    if (c === void 0) { c = {}; }
    return (tslib_1.__assign({ changeDetection: ChangeDetectionStrategy.Default }, c));
}, Directive, undefined, function (type, meta) { return (R3_COMPILE_COMPONENT || (function () { }))(type, meta); });
/**
 *
 *
 * @Annotation
 */
export var Pipe = makeDecorator('Pipe', function (p) { return (tslib_1.__assign({ pure: true }, p)); }, undefined, undefined, function (type, meta) { return (R3_COMPILE_PIPE || (function () { }))(type, meta); });
/**
 *
 * @Annotation
 */
export var Input = makePropDecorator('Input', function (bindingPropertyName) { return ({ bindingPropertyName: bindingPropertyName }); });
/**
 *
 * @Annotation
 */
export var Output = makePropDecorator('Output', function (bindingPropertyName) { return ({ bindingPropertyName: bindingPropertyName }); });
/**
 *
 * @Annotation
 */
export var HostBinding = makePropDecorator('HostBinding', function (hostPropertyName) { return ({ hostPropertyName: hostPropertyName }); });
/**
 * Binds a CSS event to a host listener and supplies configuration metadata.
 * Angular invokes the supplied handler method when the host element emits the specified event,
 * and updates the bound element with the result.
 * If the handler method returns false, applies `preventDefault` on the bound element.
 *
 * @usageNotes
 *
 * The following example declares a directive
 * that attaches a click listener to a button and counts clicks.
 *
 * ```
 * @Directive({selector: 'button[counting]'})
 * class CountClicks {
 *   numberOfClicks = 0;
 *
 *   @HostListener('click', ['$event.target'])
 *   onClick(btn) {
 *     console.log('button', btn, 'number of clicks:', this.numberOfClicks++);
 *  }
 * }
 *
 * @Component({
 *   selector: 'app',
 *   template: '<button counting>Increment</button>',
 * })
 * class App {}
 * ```
 *
 * @Annotation
 */
export var HostListener = makePropDecorator('HostListener', function (eventName, args) { return ({ eventName: eventName, args: args }); });

//# sourceMappingURL=data:application/json;base64,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