/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes,extraRequire,uselessCode} checked by tsc
 */
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { R3_COMPILE_COMPONENT, R3_COMPILE_DIRECTIVE, R3_COMPILE_PIPE } from '../ivy_switch';
import { makeDecorator, makePropDecorator } from '../util/decorators';
/**
 * Type of the Directive decorator / constructor function.
 * @record
 */
export function DirectiveDecorator() { }
/** *
 * Type of the Directive metadata.
  @type {?} */
export const Directive = makeDecorator('Directive', (dir = {}) => dir, undefined, undefined, (type, meta) => (R3_COMPILE_DIRECTIVE || (() => { }))(type, meta));
/**
 * Component decorator interface
 *
 * @record
 */
export function ComponentDecorator() { }
/** *
 * Component decorator and metadata.
 *
 * \@usageNotes
 *
 * ### Using animations
 *
 * The following snippet shows an animation trigger in a component's
 * metadata. The trigger is attached to an element in the component's
 * template, using "\@_trigger_name_", and a state expression that is evaluated
 * at run time to determine whether the animation should start.
 *
 * ```typescript
 * \@Component({
 *   selector: 'animation-cmp',
 *   templateUrl: 'animation-cmp.html',
 *   animations: [
 *     trigger('myTriggerName', [
 *       state('on', style({ opacity: 1 }),
 *       state('off', style({ opacity: 0 }),
 *       transition('on => off', [
 *         animate("1s")
 *       ])
 *     ])
 *   ]
 * })
 * ```
 *
 * ```html
 * <!-- animation-cmp.html -->
 * <div \@myTriggerName="expression">...</div>
 * ```
 *
 * ### Preserving whitespace
 *
 * Removing whitespace can greatly reduce AOT-generated code size, and speed up view creation.
 * As of Angular 6, default for `preserveWhitespaces` is false (whitespace is removed).
 * To change the default setting for all components in your application, set
 * the `preserveWhitespaces` option of the AOT compiler.
 *
 * Current implementation removes whitespace characters as follows:
 * - Trims all whitespaces at the beginning and the end of a template.
 * - Removes whitespace-only text nodes. For example,
 * `<button>Action 1</button>  <button>Action 2</button>` becomes
 * `<button>Action 1</button><button>Action 2</button>`.
 * - Replaces a series of whitespace characters in text nodes with a single space.
 * For example, `<span>\n some text\n</span>` becomes `<span> some text </span>`.
 * - Does NOT alter text nodes inside HTML tags such as `<pre>` or `<textarea>`,
 * where whitespace characters are significant.
 *
 * Note that these transformations can influence DOM nodes layout, although impact
 * should be minimal.
 *
 * You can override the default behavior to preserve whitespace characters
 * in certain fragments of a template. For example, you can exclude an entire
 * DOM sub-tree by using the `ngPreserveWhitespaces` attribute:
 *
 * ```html
 * <div ngPreserveWhitespaces>
 *     whitespaces are preserved here
 *     <span>    and here </span>
 * </div>
 * ```
 *
 * You can force a single space to be preserved in a text node by using `&ngsp;`,
 * which is replaced with a space character by Angular's template
 * compiler:
 *
 * ```html
 * <a>Spaces</a>&ngsp;<a>between</a>&ngsp;<a>links.</a>
 * <!-->compiled to be equivalent to:</>
 *  <a>Spaces</a> <a>between</a> <a>links.</a>
 * ```
 *
 * Note that sequences of `&ngsp;` are still collapsed to just one space character when
 * the `preserveWhitespaces` option is set to `false`.
 *
 * ```html
 * <a>before</a>&ngsp;&ngsp;&ngsp;<a>after</a>
 * <!-->compiled to be equivalent to:</>
 *  <a>Spaces</a> <a>between</a> <a>links.</a>
 * ```
 *
 * To preserve sequences of whitespace characters, use the
 * `ngPreserveWhitespaces` attribute.
 *
 * \@Annotation
  @type {?} */
export const Component = makeDecorator('Component', (c = {}) => (Object.assign({ changeDetection: ChangeDetectionStrategy.Default }, c)), Directive, undefined, (type, meta) => (R3_COMPILE_COMPONENT || (() => { }))(type, meta));
/**
 * Type of the Pipe decorator / constructor function.
 * @record
 */
export function PipeDecorator() { }
/** *
 *
 *
 * \@Annotation
  @type {?} */
export const Pipe = makeDecorator('Pipe', (p) => (Object.assign({ pure: true }, p)), undefined, undefined, (type, meta) => (R3_COMPILE_PIPE || (() => { }))(type, meta));
/**
 *
 * @record
 */
export function InputDecorator() { }
/** *
 *
 * \@Annotation
  @type {?} */
export const Input = makePropDecorator('Input', (bindingPropertyName) => ({ bindingPropertyName }));
/**
 * Type of the Output decorator / constructor function.
 * @record
 */
export function OutputDecorator() { }
/** *
 *
 * \@Annotation
  @type {?} */
export const Output = makePropDecorator('Output', (bindingPropertyName) => ({ bindingPropertyName }));
/**
 * Type of the HostBinding decorator / constructor function.
 * @record
 */
export function HostBindingDecorator() { }
/** *
 *
 * \@Annotation
  @type {?} */
export const HostBinding = makePropDecorator('HostBinding', (hostPropertyName) => ({ hostPropertyName }));
/**
 * Type of the HostListener decorator / constructor function.
 * @record
 */
export function HostListenerDecorator() { }
/** *
 * Binds a CSS event to a host listener and supplies configuration metadata.
 * Angular invokes the supplied handler method when the host element emits the specified event,
 * and updates the bound element with the result.
 * If the handler method returns false, applies `preventDefault` on the bound element.
 *
 * \@usageNotes
 *
 * The following example declares a directive
 * that attaches a click listener to a button and counts clicks.
 *
 * ```
 * \@Directive({selector: 'button[counting]'})
 * class CountClicks {
 *   numberOfClicks = 0;
 *
 * \@HostListener('click', ['$event.target'])
 *   onClick(btn) {
 *     console.log('button', btn, 'number of clicks:', this.numberOfClicks++);
 *  }
 * }
 *
 * \@Component({
 *   selector: 'app',
 *   template: '<button counting>Increment</button>',
 * })
 * class App {}
 * ```
 *
 * \@Annotation
  @type {?} */
export const HostListener = makePropDecorator('HostListener', (eventName, args) => ({ eventName, args }));

//# sourceMappingURL=data:application/json;base64,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