/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("@angular/compiler-cli/src/ngtsc/metadata/src/resolver", ["require", "exports", "tslib", "@angular/compiler", "path", "typescript"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    var tslib_1 = require("tslib");
    /**
     * resolver.ts implements partial computation of expressions, resolving expressions to static
     * values where possible and returning a `DynamicValue` signal when not.
     */
    var compiler_1 = require("@angular/compiler");
    var path = require("path");
    var ts = require("typescript");
    var TS_DTS_EXTENSION = /(\.d)?\.ts$/;
    /**
     * Represents a value which cannot be determined statically.
     *
     * Use `isDynamicValue` to determine whether a `ResolvedValue` is a `DynamicValue`.
     */
    var DynamicValue = /** @class */ (function () {
        function DynamicValue() {
            /**
             * This is needed so the "is DynamicValue" assertion of `isDynamicValue` actually has meaning.
             *
             * Otherwise, "is DynamicValue" is akin to "is {}" which doesn't trigger narrowing.
             */
            this._isDynamic = true;
        }
        return DynamicValue;
    }());
    exports.DynamicValue = DynamicValue;
    /**
     * An internal flyweight for `DynamicValue`. Eventually the dynamic value will carry information
     * on the location of the node that could not be statically computed.
     */
    var DYNAMIC_VALUE = new DynamicValue();
    /**
     * Used to test whether a `ResolvedValue` is a `DynamicValue`.
     */
    function isDynamicValue(value) {
        return value === DYNAMIC_VALUE;
    }
    exports.isDynamicValue = isDynamicValue;
    /**
     * A reference to a `ts.Node`.
     *
     * For example, if an expression evaluates to a function or class definition, it will be returned
     * as a `Reference` (assuming references are allowed in evaluation).
     */
    var Reference = /** @class */ (function () {
        function Reference(node) {
            this.node = node;
        }
        return Reference;
    }());
    exports.Reference = Reference;
    /**
     * A reference to a node only, without any ability to get an `Expression` representing that node.
     *
     * This is used for returning references to things like method declarations, which are not directly
     * referenceable.
     */
    var NodeReference = /** @class */ (function (_super) {
        tslib_1.__extends(NodeReference, _super);
        function NodeReference(node, moduleName) {
            var _this = _super.call(this, node) || this;
            _this.moduleName = moduleName;
            return _this;
        }
        NodeReference.prototype.toExpression = function (context) { return null; };
        NodeReference.prototype.withIdentifier = function (identifier) { return this; };
        return NodeReference;
    }(Reference));
    exports.NodeReference = NodeReference;
    /**
     * A reference to a node which has a `ts.Identifier` and can be resolved to an `Expression`.
     *
     * Imports generated by `ResolvedReference`s are always relative.
     */
    var ResolvedReference = /** @class */ (function (_super) {
        tslib_1.__extends(ResolvedReference, _super);
        function ResolvedReference(node, identifier) {
            var _this = _super.call(this, node) || this;
            _this.identifier = identifier;
            _this.expressable = true;
            return _this;
        }
        ResolvedReference.prototype.toExpression = function (context) {
            if (ts.getOriginalNode(context) === ts.getOriginalNode(this.identifier).getSourceFile()) {
                return new compiler_1.WrappedNodeExpr(this.identifier);
            }
            else {
                // Relative import from context -> this.node.getSourceFile().
                // TODO(alxhub): investigate the impact of multiple source roots here.
                // TODO(alxhub): investigate the need to map such paths via the Host for proper g3 support.
                var relative = path.posix.relative(path.dirname(context.fileName), this.node.getSourceFile().fileName)
                    .replace(TS_DTS_EXTENSION, '');
                // path.relative() does not include the leading './'.
                if (!relative.startsWith('.')) {
                    relative = "./" + relative;
                }
                // path.relative() returns the empty string (converted to './' above) if the two paths are the
                // same.
                if (relative === './') {
                    // Same file after all.
                    return new compiler_1.WrappedNodeExpr(this.identifier);
                }
                else {
                    return new compiler_1.ExternalExpr(new compiler_1.ExternalReference(relative, this.identifier.text));
                }
            }
        };
        ResolvedReference.prototype.withIdentifier = function (identifier) {
            return new ResolvedReference(this.node, identifier);
        };
        return ResolvedReference;
    }(Reference));
    exports.ResolvedReference = ResolvedReference;
    /**
     * A reference to a node which has a `ts.Identifer` and an expected absolute module name.
     *
     * An `AbsoluteReference` can be resolved to an `Expression`, and if that expression is an import
     * the module specifier will be an absolute module name, not a relative path.
     */
    var AbsoluteReference = /** @class */ (function (_super) {
        tslib_1.__extends(AbsoluteReference, _super);
        function AbsoluteReference(node, identifier, moduleName, symbolName) {
            var _this = _super.call(this, node) || this;
            _this.identifier = identifier;
            _this.moduleName = moduleName;
            _this.symbolName = symbolName;
            _this.expressable = true;
            return _this;
        }
        AbsoluteReference.prototype.toExpression = function (context) {
            if (ts.getOriginalNode(context) === ts.getOriginalNode(this.identifier).getSourceFile()) {
                return new compiler_1.WrappedNodeExpr(this.identifier);
            }
            else {
                return new compiler_1.ExternalExpr(new compiler_1.ExternalReference(this.moduleName, this.symbolName));
            }
        };
        AbsoluteReference.prototype.withIdentifier = function (identifier) {
            return new AbsoluteReference(this.node, identifier, this.moduleName, this.symbolName);
        };
        return AbsoluteReference;
    }(Reference));
    exports.AbsoluteReference = AbsoluteReference;
    /**
     * Statically resolve the given `ts.Expression` into a `ResolvedValue`.
     *
     * @param node the expression to statically resolve if possible
     * @param checker a `ts.TypeChecker` used to understand the expression
     * @param foreignFunctionResolver a function which will be used whenever a "foreign function" is
     * encountered. A foreign function is a function which has no body - usually the result of calling
     * a function declared in another library's .d.ts file. In these cases, the foreignFunctionResolver
     * will be called with the function's declaration, and can optionally return a `ts.Expression`
     * (possibly extracted from the foreign function's type signature) which will be used as the result
     * of the call.
     * @returns a `ResolvedValue` representing the resolved value
     */
    function staticallyResolve(node, host, checker, foreignFunctionResolver) {
        return new StaticInterpreter(host, checker).visit(node, {
            absoluteModuleName: null,
            scope: new Map(), foreignFunctionResolver: foreignFunctionResolver,
        });
    }
    exports.staticallyResolve = staticallyResolve;
    function literalBinaryOp(op) {
        return { op: op, literal: true };
    }
    function referenceBinaryOp(op) {
        return { op: op, literal: false };
    }
    var BINARY_OPERATORS = new Map([
        [ts.SyntaxKind.PlusToken, literalBinaryOp(function (a, b) { return a + b; })],
        [ts.SyntaxKind.MinusToken, literalBinaryOp(function (a, b) { return a - b; })],
        [ts.SyntaxKind.AsteriskToken, literalBinaryOp(function (a, b) { return a * b; })],
        [ts.SyntaxKind.SlashToken, literalBinaryOp(function (a, b) { return a / b; })],
        [ts.SyntaxKind.PercentToken, literalBinaryOp(function (a, b) { return a % b; })],
        [ts.SyntaxKind.AmpersandToken, literalBinaryOp(function (a, b) { return a & b; })],
        [ts.SyntaxKind.BarToken, literalBinaryOp(function (a, b) { return a | b; })],
        [ts.SyntaxKind.CaretToken, literalBinaryOp(function (a, b) { return a ^ b; })],
        [ts.SyntaxKind.LessThanToken, literalBinaryOp(function (a, b) { return a < b; })],
        [ts.SyntaxKind.LessThanEqualsToken, literalBinaryOp(function (a, b) { return a <= b; })],
        [ts.SyntaxKind.GreaterThanToken, literalBinaryOp(function (a, b) { return a > b; })],
        [ts.SyntaxKind.GreaterThanEqualsToken, literalBinaryOp(function (a, b) { return a >= b; })],
        [ts.SyntaxKind.LessThanLessThanToken, literalBinaryOp(function (a, b) { return a << b; })],
        [ts.SyntaxKind.GreaterThanGreaterThanToken, literalBinaryOp(function (a, b) { return a >> b; })],
        [ts.SyntaxKind.GreaterThanGreaterThanGreaterThanToken, literalBinaryOp(function (a, b) { return a >>> b; })],
        [ts.SyntaxKind.AsteriskAsteriskToken, literalBinaryOp(function (a, b) { return Math.pow(a, b); })],
        [ts.SyntaxKind.AmpersandAmpersandToken, referenceBinaryOp(function (a, b) { return a && b; })],
        [ts.SyntaxKind.BarBarToken, referenceBinaryOp(function (a, b) { return a || b; })]
    ]);
    var UNARY_OPERATORS = new Map([
        [ts.SyntaxKind.TildeToken, function (a) { return ~a; }], [ts.SyntaxKind.MinusToken, function (a) { return -a; }],
        [ts.SyntaxKind.PlusToken, function (a) { return +a; }], [ts.SyntaxKind.ExclamationToken, function (a) { return !a; }]
    ]);
    var StaticInterpreter = /** @class */ (function () {
        function StaticInterpreter(host, checker) {
            this.host = host;
            this.checker = checker;
        }
        StaticInterpreter.prototype.visit = function (node, context) {
            return this.visitExpression(node, context);
        };
        StaticInterpreter.prototype.visitExpression = function (node, context) {
            if (node.kind === ts.SyntaxKind.TrueKeyword) {
                return true;
            }
            else if (node.kind === ts.SyntaxKind.FalseKeyword) {
                return false;
            }
            else if (ts.isStringLiteral(node)) {
                return node.text;
            }
            else if (ts.isNoSubstitutionTemplateLiteral(node)) {
                return node.text;
            }
            else if (ts.isTemplateExpression(node)) {
                return this.visitTemplateExpression(node, context);
            }
            else if (ts.isNumericLiteral(node)) {
                return parseFloat(node.text);
            }
            else if (ts.isObjectLiteralExpression(node)) {
                return this.visitObjectLiteralExpression(node, context);
            }
            else if (ts.isIdentifier(node)) {
                return this.visitIdentifier(node, context);
            }
            else if (ts.isPropertyAccessExpression(node)) {
                return this.visitPropertyAccessExpression(node, context);
            }
            else if (ts.isCallExpression(node)) {
                return this.visitCallExpression(node, context);
            }
            else if (ts.isConditionalExpression(node)) {
                return this.visitConditionalExpression(node, context);
            }
            else if (ts.isPrefixUnaryExpression(node)) {
                return this.visitPrefixUnaryExpression(node, context);
            }
            else if (ts.isBinaryExpression(node)) {
                return this.visitBinaryExpression(node, context);
            }
            else if (ts.isArrayLiteralExpression(node)) {
                return this.visitArrayLiteralExpression(node, context);
            }
            else if (ts.isParenthesizedExpression(node)) {
                return this.visitParenthesizedExpression(node, context);
            }
            else if (ts.isElementAccessExpression(node)) {
                return this.visitElementAccessExpression(node, context);
            }
            else if (ts.isAsExpression(node)) {
                return this.visitExpression(node.expression, context);
            }
            else if (ts.isNonNullExpression(node)) {
                return this.visitExpression(node.expression, context);
            }
            else if (isPossibleClassDeclaration(node) && this.host.isClass(node)) {
                return this.visitDeclaration(node, context);
            }
            else {
                return DYNAMIC_VALUE;
            }
        };
        StaticInterpreter.prototype.visitArrayLiteralExpression = function (node, context) {
            var array = [];
            for (var i = 0; i < node.elements.length; i++) {
                var element = node.elements[i];
                if (ts.isSpreadElement(element)) {
                    var spread = this.visitExpression(element.expression, context);
                    if (isDynamicValue(spread)) {
                        return DYNAMIC_VALUE;
                    }
                    if (!Array.isArray(spread)) {
                        throw new Error("Unexpected value in spread expression: " + spread);
                    }
                    array.push.apply(array, tslib_1.__spread(spread));
                }
                else {
                    var result = this.visitExpression(element, context);
                    if (isDynamicValue(result)) {
                        return DYNAMIC_VALUE;
                    }
                    array.push(result);
                }
            }
            return array;
        };
        StaticInterpreter.prototype.visitObjectLiteralExpression = function (node, context) {
            var map = new Map();
            for (var i = 0; i < node.properties.length; i++) {
                var property = node.properties[i];
                if (ts.isPropertyAssignment(property)) {
                    var name_1 = this.stringNameFromPropertyName(property.name, context);
                    // Check whether the name can be determined statically.
                    if (name_1 === undefined) {
                        return DYNAMIC_VALUE;
                    }
                    map.set(name_1, this.visitExpression(property.initializer, context));
                }
                else if (ts.isShorthandPropertyAssignment(property)) {
                    var symbol = this.checker.getShorthandAssignmentValueSymbol(property);
                    if (symbol === undefined || symbol.valueDeclaration === undefined) {
                        return DYNAMIC_VALUE;
                    }
                    map.set(property.name.text, this.visitDeclaration(symbol.valueDeclaration, context));
                }
                else if (ts.isSpreadAssignment(property)) {
                    var spread = this.visitExpression(property.expression, context);
                    if (isDynamicValue(spread)) {
                        return DYNAMIC_VALUE;
                    }
                    if (!(spread instanceof Map)) {
                        throw new Error("Unexpected value in spread assignment: " + spread);
                    }
                    spread.forEach(function (value, key) { return map.set(key, value); });
                }
                else {
                    return DYNAMIC_VALUE;
                }
            }
            return map;
        };
        StaticInterpreter.prototype.visitTemplateExpression = function (node, context) {
            var pieces = [node.head.text];
            for (var i = 0; i < node.templateSpans.length; i++) {
                var span = node.templateSpans[i];
                var value = this.visit(span.expression, context);
                if (typeof value === 'string' || typeof value === 'number' || typeof value === 'boolean' ||
                    value == null) {
                    pieces.push("" + value);
                }
                else {
                    return DYNAMIC_VALUE;
                }
                pieces.push(span.literal.text);
            }
            return pieces.join('');
        };
        StaticInterpreter.prototype.visitIdentifier = function (node, context) {
            var decl = this.host.getDeclarationOfIdentifier(node);
            if (decl === null) {
                return DYNAMIC_VALUE;
            }
            var result = this.visitDeclaration(decl.node, tslib_1.__assign({}, context, { absoluteModuleName: decl.viaModule || context.absoluteModuleName }));
            if (result instanceof Reference) {
                return result.withIdentifier(node);
            }
            else {
                return result;
            }
        };
        StaticInterpreter.prototype.visitDeclaration = function (node, context) {
            if (this.host.isClass(node)) {
                return this.getReference(node, context);
            }
            else if (ts.isVariableDeclaration(node)) {
                if (!node.initializer) {
                    return undefined;
                }
                return this.visitExpression(node.initializer, context);
            }
            else if (ts.isParameter(node) && context.scope.has(node)) {
                return context.scope.get(node);
            }
            else if (ts.isExportAssignment(node)) {
                return this.visitExpression(node.expression, context);
            }
            else if (ts.isSourceFile(node)) {
                return this.visitSourceFile(node, context);
            }
            else {
                return this.getReference(node, context);
            }
        };
        StaticInterpreter.prototype.visitElementAccessExpression = function (node, context) {
            var lhs = this.visitExpression(node.expression, context);
            if (node.argumentExpression === undefined) {
                throw new Error("Expected argument in ElementAccessExpression");
            }
            if (isDynamicValue(lhs)) {
                return DYNAMIC_VALUE;
            }
            var rhs = this.visitExpression(node.argumentExpression, context);
            if (isDynamicValue(rhs)) {
                return DYNAMIC_VALUE;
            }
            if (typeof rhs !== 'string' && typeof rhs !== 'number') {
                throw new Error("ElementAccessExpression index should be string or number, got " + typeof rhs + ": " + rhs);
            }
            return this.accessHelper(lhs, rhs, context);
        };
        StaticInterpreter.prototype.visitPropertyAccessExpression = function (node, context) {
            var lhs = this.visitExpression(node.expression, context);
            var rhs = node.name.text;
            // TODO: handle reference to class declaration.
            if (isDynamicValue(lhs)) {
                return DYNAMIC_VALUE;
            }
            return this.accessHelper(lhs, rhs, context);
        };
        StaticInterpreter.prototype.visitSourceFile = function (node, context) {
            var _this = this;
            var declarations = this.host.getExportsOfModule(node);
            if (declarations === null) {
                return DYNAMIC_VALUE;
            }
            var map = new Map();
            declarations.forEach(function (decl, name) {
                var value = _this.visitDeclaration(decl.node, tslib_1.__assign({}, context, { absoluteModuleName: decl.viaModule || context.absoluteModuleName }));
                map.set(name, value);
            });
            return map;
        };
        StaticInterpreter.prototype.accessHelper = function (lhs, rhs, context) {
            var strIndex = "" + rhs;
            if (lhs instanceof Map) {
                if (lhs.has(strIndex)) {
                    return lhs.get(strIndex);
                }
                else {
                    throw new Error("Invalid map access: [" + Array.from(lhs.keys()) + "] dot " + rhs);
                }
            }
            else if (Array.isArray(lhs)) {
                if (rhs === 'length') {
                    return lhs.length;
                }
                if (typeof rhs !== 'number' || !Number.isInteger(rhs)) {
                    return DYNAMIC_VALUE;
                }
                if (rhs < 0 || rhs >= lhs.length) {
                    throw new Error("Index out of bounds: " + rhs + " vs " + lhs.length);
                }
                return lhs[rhs];
            }
            else if (lhs instanceof Reference) {
                var ref = lhs.node;
                if (isPossibleClassDeclaration(ref) && this.host.isClass(ref)) {
                    var absoluteModuleName = context.absoluteModuleName;
                    if (lhs instanceof NodeReference || lhs instanceof AbsoluteReference) {
                        absoluteModuleName = lhs.moduleName || absoluteModuleName;
                    }
                    var value = undefined;
                    var member = this.host.getMembersOfClass(ref).find(function (member) { return member.isStatic && member.name === strIndex; });
                    if (member !== undefined) {
                        if (member.value !== null) {
                            value = this.visitExpression(member.value, context);
                        }
                        else if (member.implementation !== null) {
                            value = new NodeReference(member.implementation, absoluteModuleName);
                        }
                        else {
                            value = new NodeReference(member.node, absoluteModuleName);
                        }
                    }
                    return value;
                }
            }
            throw new Error("Invalid dot property access: " + lhs + " dot " + rhs);
        };
        StaticInterpreter.prototype.visitCallExpression = function (node, context) {
            var _this = this;
            var lhs = this.visitExpression(node.expression, context);
            if (!(lhs instanceof Reference)) {
                throw new Error("attempting to call something that is not a function: " + lhs);
            }
            else if (!isFunctionOrMethodDeclaration(lhs.node)) {
                throw new Error("calling something that is not a function declaration? " + ts.SyntaxKind[lhs.node.kind]);
            }
            var fn = lhs.node;
            // If the function is foreign (declared through a d.ts file), attempt to resolve it with the
            // foreignFunctionResolver, if one is specified.
            if (fn.body === undefined) {
                var expr = null;
                if (context.foreignFunctionResolver) {
                    expr = context.foreignFunctionResolver(fn);
                }
                if (expr === null) {
                    throw new Error("could not resolve foreign function declaration");
                }
                // If the function is declared in a different file, resolve the foreign function expression
                // using the absolute module name of that file (if any).
                var absoluteModuleName = context.absoluteModuleName;
                if (lhs instanceof NodeReference || lhs instanceof AbsoluteReference) {
                    absoluteModuleName = lhs.moduleName || absoluteModuleName;
                }
                return this.visitExpression(expr, tslib_1.__assign({}, context, { absoluteModuleName: absoluteModuleName }));
            }
            var body = fn.body;
            if (body.statements.length !== 1 || !ts.isReturnStatement(body.statements[0])) {
                throw new Error('Function body must have a single return statement only.');
            }
            var ret = body.statements[0];
            var newScope = new Map();
            fn.parameters.forEach(function (param, index) {
                var value = undefined;
                if (index < node.arguments.length) {
                    var arg = node.arguments[index];
                    value = _this.visitExpression(arg, context);
                }
                if (value === undefined && param.initializer !== undefined) {
                    value = _this.visitExpression(param.initializer, context);
                }
                newScope.set(param, value);
            });
            return ret.expression !== undefined ?
                this.visitExpression(ret.expression, tslib_1.__assign({}, context, { scope: newScope })) :
                undefined;
        };
        StaticInterpreter.prototype.visitConditionalExpression = function (node, context) {
            var condition = this.visitExpression(node.condition, context);
            if (isDynamicValue(condition)) {
                return condition;
            }
            if (condition) {
                return this.visitExpression(node.whenTrue, context);
            }
            else {
                return this.visitExpression(node.whenFalse, context);
            }
        };
        StaticInterpreter.prototype.visitPrefixUnaryExpression = function (node, context) {
            var operatorKind = node.operator;
            if (!UNARY_OPERATORS.has(operatorKind)) {
                throw new Error("Unsupported prefix unary operator: " + ts.SyntaxKind[operatorKind]);
            }
            var op = UNARY_OPERATORS.get(operatorKind);
            var value = this.visitExpression(node.operand, context);
            return isDynamicValue(value) ? DYNAMIC_VALUE : op(value);
        };
        StaticInterpreter.prototype.visitBinaryExpression = function (node, context) {
            var tokenKind = node.operatorToken.kind;
            if (!BINARY_OPERATORS.has(tokenKind)) {
                throw new Error("Unsupported binary operator: " + ts.SyntaxKind[tokenKind]);
            }
            var opRecord = BINARY_OPERATORS.get(tokenKind);
            var lhs, rhs;
            if (opRecord.literal) {
                lhs = literal(this.visitExpression(node.left, context));
                rhs = literal(this.visitExpression(node.right, context));
            }
            else {
                lhs = this.visitExpression(node.left, context);
                rhs = this.visitExpression(node.right, context);
            }
            return isDynamicValue(lhs) || isDynamicValue(rhs) ? DYNAMIC_VALUE : opRecord.op(lhs, rhs);
        };
        StaticInterpreter.prototype.visitParenthesizedExpression = function (node, context) {
            return this.visitExpression(node.expression, context);
        };
        StaticInterpreter.prototype.stringNameFromPropertyName = function (node, context) {
            if (ts.isIdentifier(node) || ts.isStringLiteral(node) || ts.isNumericLiteral(node)) {
                return node.text;
            }
            else { // ts.ComputedPropertyName
                var literal_1 = this.visitExpression(node.expression, context);
                return typeof literal_1 === 'string' ? literal_1 : undefined;
            }
        };
        StaticInterpreter.prototype.getReference = function (node, context) {
            var id = identifierOfDeclaration(node);
            if (id === undefined) {
                throw new Error("Don't know how to refer to " + ts.SyntaxKind[node.kind]);
            }
            if (context.absoluteModuleName !== null) {
                // TODO(alxhub): investigate whether this can get symbol names wrong in the event of
                // re-exports under different names.
                return new AbsoluteReference(node, id, context.absoluteModuleName, id.text);
            }
            else {
                return new ResolvedReference(node, id);
            }
        };
        return StaticInterpreter;
    }());
    function isFunctionOrMethodDeclaration(node) {
        return ts.isFunctionDeclaration(node) || ts.isMethodDeclaration(node);
    }
    function literal(value) {
        if (value === null || value === undefined || typeof value === 'string' ||
            typeof value === 'number' || typeof value === 'boolean') {
            return value;
        }
        if (isDynamicValue(value)) {
            return DYNAMIC_VALUE;
        }
        throw new Error("Value " + value + " is not literal and cannot be used in this context.");
    }
    function identifierOfDeclaration(decl) {
        if (ts.isClassDeclaration(decl)) {
            return decl.name;
        }
        else if (ts.isFunctionDeclaration(decl)) {
            return decl.name;
        }
        else if (ts.isVariableDeclaration(decl) && ts.isIdentifier(decl.name)) {
            return decl.name;
        }
        else if (ts.isShorthandPropertyAssignment(decl)) {
            return decl.name;
        }
        else {
            return undefined;
        }
    }
    function isPossibleClassDeclaration(node) {
        return ts.isClassDeclaration(node) || ts.isVariableDeclaration(node);
    }
});
//# sourceMappingURL=data:application/json;base64,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