"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const rxjs_1 = require("rxjs");
class Logger extends rxjs_1.Observable {
    constructor(name, parent = null) {
        super();
        this.name = name;
        this.parent = parent;
        this._subject = new rxjs_1.Subject();
        const path = [];
        let p = parent;
        while (p) {
            path.push(p.name);
            p = p.parent;
        }
        this._metadata = { name, path };
        this._observable = this._subject.asObservable();
        if (this.parent && this.parent._subject) {
            // When the parent completes, complete us as well.
            this.parent._subject.subscribe(undefined, undefined, () => this.complete());
        }
    }
    get _observable() { return this._obs; }
    set _observable(v) {
        if (this._subscription) {
            this._subscription.unsubscribe();
        }
        this._obs = v;
        if (this.parent) {
            this._subscription = this.subscribe((value) => {
                if (this.parent) {
                    this.parent._subject.next(value);
                }
            }, (error) => {
                if (this.parent) {
                    this.parent._subject.error(error);
                }
            }, () => {
                if (this._subscription) {
                    this._subscription.unsubscribe();
                }
                this._subscription = null;
            });
        }
    }
    asApi() {
        return {
            createChild: (name) => this.createChild(name),
            log: (level, message, metadata) => {
                return this.log(level, message, metadata);
            },
            debug: (message, metadata) => this.debug(message, metadata),
            info: (message, metadata) => this.info(message, metadata),
            warn: (message, metadata) => this.warn(message, metadata),
            error: (message, metadata) => this.error(message, metadata),
            fatal: (message, metadata) => this.fatal(message, metadata),
        };
    }
    createChild(name) {
        return new this.constructor(name, this);
    }
    complete() {
        this._subject.complete();
    }
    log(level, message, metadata = {}) {
        const entry = Object.assign({}, this._metadata, metadata, {
            level, message, timestamp: +Date.now(),
        });
        this._subject.next(entry);
    }
    debug(message, metadata = {}) {
        return this.log('debug', message, metadata);
    }
    info(message, metadata = {}) {
        return this.log('info', message, metadata);
    }
    warn(message, metadata = {}) {
        return this.log('warn', message, metadata);
    }
    error(message, metadata = {}) {
        return this.log('error', message, metadata);
    }
    fatal(message, metadata = {}) {
        return this.log('fatal', message, metadata);
    }
    toString() {
        return `<Logger(${this.name})>`;
    }
    lift(operator) {
        return this._observable.lift(operator);
    }
    subscribe(_observerOrNext, _error, _complete) {
        return this._observable.subscribe.apply(this._observable, arguments);
    }
    forEach(next, PromiseCtor) {
        return this._observable.forEach(next, PromiseCtor);
    }
}
exports.Logger = Logger;
//# sourceMappingURL=data:application/json;base64,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