"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const interface_1 = require("../interface");
const allTypes = ['string', 'integer', 'number', 'object', 'array', 'boolean', 'null'];
function findTypes(schema) {
    if (!schema) {
        return new Set();
    }
    let potentials;
    if (typeof schema.type === 'string') {
        potentials = new Set([schema.type]);
    }
    else if (Array.isArray(schema.type)) {
        potentials = new Set(schema.type);
    }
    else {
        potentials = new Set(allTypes);
    }
    if (interface_1.isJsonObject(schema.not)) {
        const notTypes = findTypes(schema.not);
        potentials = new Set([...potentials].filter(p => !notTypes.has(p)));
    }
    if (Array.isArray(schema.allOf)) {
        for (const sub of schema.allOf) {
            const types = findTypes(sub);
            potentials = new Set([...potentials].filter(p => types.has(p)));
        }
    }
    if (Array.isArray(schema.oneOf)) {
        let options = new Set();
        for (const sub of schema.oneOf) {
            const types = findTypes(sub);
            options = new Set([...options, ...types]);
        }
        potentials = new Set([...potentials].filter(p => options.has(p)));
    }
    if (Array.isArray(schema.anyOf)) {
        let options = new Set();
        for (const sub of schema.anyOf) {
            const types = findTypes(sub);
            options = new Set([...options, ...types]);
        }
        potentials = new Set([...potentials].filter(p => options.has(p)));
    }
    return potentials;
}
function addUndefinedDefaults(value, _pointer, schema) {
    if (!schema) {
        return value;
    }
    const types = findTypes(schema);
    if (types.size === 0) {
        return value;
    }
    let type;
    if (types.size === 1) {
        // only one potential type
        type = Array.from(types)[0];
    }
    else if (types.size === 2 && types.has('array') && types.has('object')) {
        // need to create one of them and array is simpler
        type = 'array';
    }
    else if (schema.properties && types.has('object')) {
        // assume object
        type = 'object';
    }
    else if (schema.items && types.has('array')) {
        // assume array
        type = 'array';
    }
    else {
        // anything else needs to be checked by the consumer anyway
        return value;
    }
    if (type === 'array') {
        return value == undefined ? [] : value;
    }
    if (type === 'object') {
        let newValue;
        if (value == undefined) {
            newValue = {};
        }
        else if (interface_1.isJsonObject(value)) {
            newValue = value;
        }
        else {
            return value;
        }
        if (!interface_1.isJsonObject(schema.properties)) {
            return newValue;
        }
        for (const propName of Object.getOwnPropertyNames(schema.properties)) {
            if (propName in newValue) {
                continue;
            }
            // TODO: Does not currently handle more complex schemas (oneOf/anyOf/etc.)
            const defaultValue = schema.properties[propName].default;
            newValue[propName] = defaultValue;
        }
        return newValue;
    }
    return value;
}
exports.addUndefinedDefaults = addUndefinedDefaults;
//# sourceMappingURL=data:application/json;base64,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