"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const src_1 = require("../src");
/**
 * An EngineHost that support multiple hosts in a fallback configuration. If a host does not
 * have a collection/schematics, use the following host before giving up.
 */
class FallbackEngineHost {
    constructor() {
        this._hosts = [];
    }
    addHost(host) {
        this._hosts.push(host);
    }
    createCollectionDescription(name) {
        for (const host of this._hosts) {
            try {
                const description = host.createCollectionDescription(name);
                return { name, host, description };
            }
            catch (_) {
            }
        }
        throw new src_1.UnknownCollectionException(name);
    }
    createSchematicDescription(name, collection) {
        const description = collection.host.createSchematicDescription(name, collection.description);
        if (!description) {
            return null;
        }
        return { name, collection, description };
    }
    getSchematicRuleFactory(schematic, collection) {
        return collection.host.getSchematicRuleFactory(schematic.description, collection.description);
    }
    createSourceFromUrl(url, context) {
        return context.schematic.collection.description.host.createSourceFromUrl(url, context);
    }
    transformOptions(schematic, options) {
        return (rxjs_1.of(options)
            .pipe(...this._hosts.map(host => operators_1.mergeMap(opt => host.transformOptions(schematic, opt)))));
    }
    transformContext(context) {
        let result = context;
        this._hosts.forEach(host => {
            result = (host.transformContext(result) || result);
        });
        return result;
    }
    /**
     * @deprecated Use `listSchematicNames`.
     */
    listSchematics(collection) {
        return this.listSchematicNames(collection.description);
    }
    listSchematicNames(collection) {
        const allNames = new Set();
        this._hosts.forEach(host => {
            try {
                host.listSchematicNames(collection.description).forEach(name => allNames.add(name));
            }
            catch (_) { }
        });
        return [...allNames];
    }
    createTaskExecutor(name) {
        for (const host of this._hosts) {
            if (host.hasTaskExecutor(name)) {
                return host.createTaskExecutor(name);
            }
        }
        return rxjs_1.throwError(new src_1.UnregisteredTaskException(name));
    }
    hasTaskExecutor(name) {
        for (const host of this._hosts) {
            if (host.hasTaskExecutor(name)) {
                return true;
            }
        }
        return false;
    }
}
exports.FallbackEngineHost = FallbackEngineHost;
//# sourceMappingURL=data:application/json;base64,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