"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const src_1 = require("../src");
const call_1 = require("../src/rules/call");
const node_1 = require("../tasks/node");
const tools_1 = require("../tools");
class UnitTestTree extends src_1.DelegateTree {
    get files() {
        const result = [];
        this.visit(path => result.push(path));
        return result;
    }
    readContent(path) {
        const buffer = this.read(path);
        if (buffer === null) {
            return '';
        }
        return buffer.toString();
    }
}
exports.UnitTestTree = UnitTestTree;
class SchematicTestRunner {
    constructor(_collectionName, collectionPath) {
        this._collectionName = _collectionName;
        this._engineHost = new tools_1.NodeModulesTestEngineHost();
        this._engine = new src_1.SchematicEngine(this._engineHost);
        this._engineHost.registerCollection(_collectionName, collectionPath);
        this._logger = new core_1.logging.Logger('test');
        const registry = new core_1.schema.CoreSchemaRegistry(src_1.formats.standardFormats);
        this._engineHost.registerOptionsTransform(tools_1.validateOptionsWithSchema(registry));
        this._engineHost.registerTaskExecutor(node_1.BuiltinTaskExecutor.NodePackage);
        this._engineHost.registerTaskExecutor(node_1.BuiltinTaskExecutor.RepositoryInitializer);
        this._engineHost.registerTaskExecutor(node_1.BuiltinTaskExecutor.RunSchematic);
        this._engineHost.registerTaskExecutor(node_1.BuiltinTaskExecutor.TslintFix);
        this._collection = this._engine.createCollection(this._collectionName);
    }
    get engine() { return this._engine; }
    get logger() { return this._logger; }
    get tasks() { return [...this._engineHost.tasks]; }
    runSchematicAsync(schematicName, opts, tree) {
        const schematic = this._collection.createSchematic(schematicName, true);
        const host = rxjs_1.of(tree || new src_1.HostTree);
        this._engineHost.clearTasks();
        return schematic.call(opts || {}, host, { logger: this._logger })
            .pipe(operators_1.map(tree => new UnitTestTree(tree)));
    }
    runSchematic(schematicName, opts, tree) {
        const schematic = this._collection.createSchematic(schematicName, true);
        let result = null;
        let error;
        const host = rxjs_1.of(tree || new src_1.HostTree);
        this._engineHost.clearTasks();
        schematic.call(opts || {}, host, { logger: this._logger })
            .subscribe(t => result = new UnitTestTree(t), e => error = e);
        if (error) {
            throw error;
        }
        if (result === null) {
            throw new Error('Schematic is async, please use runSchematicAsync');
        }
        return result;
    }
    runExternalSchematicAsync(collectionName, schematicName, opts, tree) {
        const externalCollection = this._engine.createCollection(collectionName);
        const schematic = externalCollection.createSchematic(schematicName, true);
        const host = rxjs_1.of(tree || new src_1.HostTree);
        this._engineHost.clearTasks();
        return schematic.call(opts || {}, host, { logger: this._logger })
            .pipe(operators_1.map(tree => new UnitTestTree(tree)));
    }
    runExternalSchematic(collectionName, schematicName, opts, tree) {
        const externalCollection = this._engine.createCollection(collectionName);
        const schematic = externalCollection.createSchematic(schematicName, true);
        let result = null;
        const host = rxjs_1.of(tree || new src_1.HostTree);
        this._engineHost.clearTasks();
        schematic.call(opts || {}, host, { logger: this._logger })
            .subscribe(t => result = new UnitTestTree(t));
        if (result === null) {
            throw new Error('Schematic is async, please use runSchematicAsync');
        }
        return result;
    }
    callRule(rule, tree, parentContext) {
        const context = this._engine.createContext({}, parentContext);
        return call_1.callRule(rule, rxjs_1.of(tree), context);
    }
}
exports.SchematicTestRunner = SchematicTestRunner;
//# sourceMappingURL=data:application/json;base64,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