"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const core_1 = require("@angular-devkit/core");
const entry_1 = require("./entry");
const virtual_1 = require("./virtual");
class FileSystemDirEntry extends virtual_1.VirtualDirEntry {
    constructor(_host, tree, path = core_1.normalize('/')) {
        super(tree, path);
        this._host = _host;
    }
    _createDir(name) {
        return new FileSystemDirEntry(this._host, this._tree, core_1.join(this._path, name));
    }
    get parent() {
        return this._path == '/' ? null : this._tree.getDir(core_1.dirname(this._path));
    }
    get subdirs() {
        const result = new Set();
        try {
            this._host.list(this._path)
                .filter(name => this._host.isDirectory(core_1.join(this._path, name)))
                .forEach(name => result.add(core_1.fragment(name)));
        }
        catch (e) {
            if (e.code != 'ENOENT' && e.code != 'ENOTDIR') {
                throw e;
            }
        }
        for (const path of this._tree.staging.keys()) {
            if (path.startsWith(this._path) && core_1.dirname(path) != this._path) {
                result.add(core_1.basename(path));
            }
        }
        return [...result];
    }
    get subfiles() {
        const result = new Set();
        try {
            this._host.list(this._path)
                .filter(name => !this._host.isDirectory(core_1.join(this._path, name)))
                .forEach(name => result.add(core_1.fragment(name)));
        }
        catch (e) {
            if (e.code != 'ENOENT' && e.code != 'ENOTDIR') {
                throw e;
            }
        }
        for (const path of this._tree.staging.keys()) {
            if (path.startsWith(this._path) && core_1.dirname(path) == this._path) {
                result.add(core_1.basename(path));
            }
        }
        return [...result];
    }
    file(name) {
        return this._tree.get(core_1.join(this._path, name));
    }
}
exports.FileSystemDirEntry = FileSystemDirEntry;
class FileSystemTree extends virtual_1.VirtualTree {
    constructor(host) {
        super();
        this._initialized = false;
        this._host = new core_1.virtualFs.SyncDelegateHost(host);
        this._root = new FileSystemDirEntry(this._host, this);
    }
    get tree() {
        const host = this._host;
        if (!this._initialized) {
            this._initialized = true;
            this._recursiveFileList().forEach(path => {
                this._tree.set(path, new entry_1.LazyFileEntry(path, () => {
                    return Buffer.from(host.read(path));
                }));
            });
        }
        return this._tree;
    }
    get(path) {
        const normalizedPath = this._normalizePath(path);
        let entry = this._cacheMap.get(normalizedPath) || this._tree.get(normalizedPath) || null;
        if (entry == null && !this._initialized) {
            const systemPath = normalizedPath;
            const fileExists = this._host.exists(systemPath) && !this._host.isDirectory(systemPath);
            if (fileExists) {
                const host = this._host;
                entry = new entry_1.LazyFileEntry(normalizedPath, () => Buffer.from(host.read(systemPath)));
                this._tree.set(normalizedPath, entry);
            }
        }
        return entry;
    }
    branch() {
        const newTree = new FileSystemTree(this._host.delegate);
        this._copyTo(newTree);
        return newTree;
    }
    _copyTo(tree) {
        if (tree instanceof FileSystemTree) {
            const x = tree;
            x._tree = this._tree;
            x._initialized = this._initialized;
            this._actions.forEach(action => x._actions.push(action));
            [...this._cacheMap.entries()].forEach(([path, entry]) => {
                x._cacheMap.set(path, entry);
            });
        }
        else {
            super._copyTo(tree);
        }
    }
    _recursiveFileList() {
        const host = this._host;
        const list = [];
        function recurse(schematicPath) {
            for (const name of host.list(schematicPath)) {
                const normalizedPath = core_1.join(schematicPath, name);
                if (host.isDirectory(normalizedPath)) {
                    recurse(normalizedPath);
                }
                else {
                    list.push(normalizedPath);
                }
            }
        }
        recurse(core_1.normalize('/'));
        return list;
    }
}
exports.FileSystemTree = FileSystemTree;
class FileSystemCreateTree extends FileSystemTree {
    constructor(host) {
        super(host);
        this._recursiveFileList().forEach(path => {
            this.create(path, Buffer.from(this._host.read(path)));
        });
        this._initialized = true;
    }
}
exports.FileSystemCreateTree = FileSystemCreateTree;
//# sourceMappingURL=data:application/json;base64,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