"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const interface_1 = require("../tree/interface");
const static_1 = require("../tree/static");
/**
 * Run a schematic from a separate collection.
 *
 * @param collectionName The name of the collection that contains the schematic to run.
 * @param schematicName The name of the schematic to run.
 * @param options The options to pass as input to the RuleFactory.
 */
function externalSchematic(collectionName, schematicName, options) {
    return (input, context) => {
        const collection = context.engine.createCollection(collectionName);
        const schematic = collection.createSchematic(schematicName);
        return schematic.call(options, rxjs_1.of(static_1.branch(input)), context);
    };
}
exports.externalSchematic = externalSchematic;
/**
 * Run a schematic from the same collection.
 *
 * @param schematicName The name of the schematic to run.
 * @param options The options to pass as input to the RuleFactory.
 */
function schematic(schematicName, options) {
    return (input, context) => {
        const collection = context.schematic.collection;
        const schematic = collection.createSchematic(schematicName, true);
        return schematic.call(options, rxjs_1.of(static_1.branch(input)), context).pipe(operators_1.last(), operators_1.map(x => {
            // We allow overwrite conflict here because they're the only merge conflict we particularly
            // don't want to deal with; the input tree might have an OVERWRITE which the sub
            input.merge(x, interface_1.MergeStrategy.AllowOverwriteConflict);
            return input;
        }));
    };
}
exports.schematic = schematic;
//# sourceMappingURL=data:application/json;base64,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