"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const _env = (typeof process == 'object' && process.env) || {};
const _platform = (typeof process == 'object' && process.platform) || '';
const _versions = (typeof process == 'object' && process.versions) || { node: '' };
const _os = (typeof os == 'object' && os) || { release: () => '' };
const streamMap = new WeakMap();
const ciVars = ['TRAVIS', 'CIRCLECI', 'APPVEYOR', 'GITLAB_CI'];
function _getColorLevel(stream) {
    if ('FORCE_COLOR' in _env) {
        if (_env.FORCE_COLOR === '1') {
            return 3;
        }
        else if (_env.FORCE_COLOR === '0') {
            return 0;
        }
    }
    if (stream && !stream.isTTY && !_env.MSYSTEM) {
        return 0;
    }
    if (_platform.startsWith('win32') && !_env.MSYSTEM) {
        // Node.js 7.5.0 is the first version of Node.js to include a patch to
        // libuv that enables 256 color output on Windows. Anything earlier and it
        // won't work. However, here we target Node.js 8 at minimum as it is an LTS
        // release, and Node.js 7 is not. Windows 10 build 10586 is the first Windows
        // release that supports 256 colors.
        const osRelease = _os.release().split('.');
        if (Number(_versions.node.split('.')[0]) >= 8
            && Number(osRelease[0]) >= 10
            && Number(osRelease[2]) >= 10586) {
            return 2;
        }
        return 1;
    }
    if ('CI' in _env) {
        if (ciVars.some(sign => sign in _env) || _env.CI_NAME === 'codeship') {
            return 1;
        }
        return 0;
    }
    if ('TEAMCITY_VERSION' in _env) {
        return /^(9\.(0*[1-9]\d*)\.|\d{2,}\.)/.test(_env.TEAMCITY_VERSION) ? 1 : 0;
    }
    if ('TERM_PROGRAM' in _env) {
        const version = parseInt((_env.TERM_PROGRAM_VERSION || '').split('.')[0], 10);
        switch (_env.TERM_PROGRAM) {
            case 'iTerm.app':
                return version >= 3 ? 3 : 2;
            case 'Hyper':
                return 3;
            case 'Apple_Terminal':
                return 2;
            // No default
        }
    }
    if (/-256(color)?$/i.test(_env.TERM)) {
        return 2;
    }
    if (/^screen|^xterm|^vt100|^rxvt|color|ansi|cygwin|linux/i.test(_env.TERM)) {
        return 1;
    }
    if ('COLORTERM' in _env) {
        return 1;
    }
    if (_env.TERM === 'dumb') {
        return 0;
    }
    return 0;
}
function _getRows() {
    return process.stdout.rows || null;
}
function _getColumns() {
    return process.stdout.columns || null;
}
function _createCapabilities(stream, isTerminalStream) {
    const level = _getColorLevel(stream);
    return {
        readable: stream.readable,
        writable: stream.writable,
        text: true,
        colors: level > 0,
        color256: level > 1,
        color16m: level > 2,
        rows: isTerminalStream ? _getRows() : null,
        columns: isTerminalStream ? _getColumns() : null,
    };
}
function getCapabilities(stream, isTerminalStream = !!stream.isTTY) {
    let maybeCaps = streamMap.get(stream);
    if (!maybeCaps) {
        maybeCaps = _createCapabilities(stream, isTerminalStream);
        streamMap.set(stream, maybeCaps);
    }
    return maybeCaps;
}
exports.getCapabilities = getCapabilities;
exports.stdin = getCapabilities(process.stdin);
exports.stdout = getCapabilities(process.stdout);
exports.stderr = getCapabilities(process.stderr);
//# sourceMappingURL=data:application/json;base64,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