"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("../path");
const buffer_1 = require("./buffer");
const memory_1 = require("./memory");
const sync_1 = require("./sync");
class TestHost extends memory_1.SimpleMemoryHost {
    constructor(map = {}) {
        super();
        this._records = [];
        for (const filePath of Object.getOwnPropertyNames(map)) {
            this._write(path_1.normalize(filePath), buffer_1.stringToFileBuffer(map[filePath]));
        }
    }
    get records() {
        return [...this._records];
    }
    clearRecords() {
        this._records = [];
    }
    get files() {
        const sync = this.sync;
        function _visit(p) {
            return sync.list(p)
                .map(fragment => path_1.join(p, fragment))
                .reduce((files, path) => {
                if (sync.isDirectory(path)) {
                    return files.concat(_visit(path));
                }
                else {
                    return files.concat(path);
                }
            }, []);
        }
        return _visit(path_1.normalize('/'));
    }
    get sync() {
        if (!this._sync) {
            this._sync = new sync_1.SyncDelegateHost(this);
        }
        return this._sync;
    }
    clone() {
        const newHost = new TestHost();
        newHost._cache = new Map(this._cache);
        return newHost;
    }
    // Override parents functions to keep a record of all operators that were done.
    _write(path, content) {
        this._records.push({ kind: 'write', path });
        return super._write(path, content);
    }
    _read(path) {
        this._records.push({ kind: 'read', path });
        return super._read(path);
    }
    _delete(path) {
        this._records.push({ kind: 'delete', path });
        return super._delete(path);
    }
    _rename(from, to) {
        this._records.push({ kind: 'rename', from, to });
        return super._rename(from, to);
    }
    _list(path) {
        this._records.push({ kind: 'list', path });
        return super._list(path);
    }
    _exists(path) {
        this._records.push({ kind: 'exists', path });
        return super._exists(path);
    }
    _isDirectory(path) {
        this._records.push({ kind: 'isDirectory', path });
        return super._isDirectory(path);
    }
    _isFile(path) {
        this._records.push({ kind: 'isFile', path });
        return super._isFile(path);
    }
    _stat(path) {
        this._records.push({ kind: 'stat', path });
        return super._stat(path);
    }
    _watch(path, options) {
        this._records.push({ kind: 'watch', path });
        return super._watch(path, options);
    }
    $write(path, content) {
        return super._write(path_1.normalize(path), buffer_1.stringToFileBuffer(content));
    }
    $read(path) {
        return buffer_1.fileBufferToString(super._read(path_1.normalize(path)));
    }
    $list(path) {
        return super._list(path_1.normalize(path));
    }
    $exists(path) {
        return super._exists(path_1.normalize(path));
    }
    $isDirectory(path) {
        return super._isDirectory(path_1.normalize(path));
    }
    $isFile(path) {
        return super._isFile(path_1.normalize(path));
    }
}
exports.TestHost = TestHost;
//# sourceMappingURL=data:application/json;base64,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