"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const rxjs_1 = require("rxjs");
class Logger extends rxjs_1.Observable {
    constructor(name, parent = null) {
        super();
        this.name = name;
        this.parent = parent;
        this._subject = new rxjs_1.Subject();
        const path = [];
        let p = parent;
        while (p) {
            path.push(p.name);
            p = p.parent;
        }
        this._metadata = { name, path };
        this._observable = this._subject.asObservable();
        if (this.parent && this.parent._subject) {
            // When the parent completes, complete us as well.
            this.parent._subject.subscribe(undefined, undefined, () => this.complete());
        }
    }
    get _observable() { return this._obs; }
    set _observable(v) {
        if (this._subscription) {
            this._subscription.unsubscribe();
        }
        this._obs = v;
        if (this.parent) {
            this._subscription = this.subscribe((value) => {
                if (this.parent) {
                    this.parent._subject.next(value);
                }
            }, (error) => {
                if (this.parent) {
                    this.parent._subject.error(error);
                }
            }, () => {
                if (this._subscription) {
                    this._subscription.unsubscribe();
                }
                this._subscription = null;
            });
        }
    }
    asApi() {
        return {
            createChild: (name) => this.createChild(name),
            log: (level, message, metadata) => {
                return this.log(level, message, metadata);
            },
            debug: (message, metadata) => this.debug(message, metadata),
            info: (message, metadata) => this.info(message, metadata),
            warn: (message, metadata) => this.warn(message, metadata),
            error: (message, metadata) => this.error(message, metadata),
            fatal: (message, metadata) => this.fatal(message, metadata),
        };
    }
    createChild(name) {
        return new this.constructor(name, this);
    }
    complete() {
        this._subject.complete();
    }
    log(level, message, metadata = {}) {
        const entry = Object.assign({}, this._metadata, metadata, {
            level, message, timestamp: +Date.now(),
        });
        this._subject.next(entry);
    }
    debug(message, metadata = {}) {
        return this.log('debug', message, metadata);
    }
    info(message, metadata = {}) {
        return this.log('info', message, metadata);
    }
    warn(message, metadata = {}) {
        return this.log('warn', message, metadata);
    }
    error(message, metadata = {}) {
        return this.log('error', message, metadata);
    }
    fatal(message, metadata = {}) {
        return this.log('fatal', message, metadata);
    }
    toString() {
        return `<Logger(${this.name})>`;
    }
    lift(operator) {
        return this._observable.lift(operator);
    }
    subscribe(_observerOrNext, _error, _complete) {
        return this._observable.subscribe.apply(this._observable, arguments);
    }
    forEach(next, PromiseCtor) {
        return this._observable.forEach(next, PromiseCtor);
    }
}
exports.Logger = Logger;
//# sourceMappingURL=data:application/json;base64,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