"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
const ts = require("typescript");
function getFoldFileTransformer(program) {
    const checker = program.getTypeChecker();
    return (context) => {
        const transformer = (sf) => {
            const classes = findClassDeclarations(sf);
            const statements = findClassStaticPropertyAssignments(sf, checker, classes);
            const visitor = (node) => {
                // Check if node is a statement to be dropped.
                if (statements.find((st) => st.expressionStatement === node)) {
                    return undefined;
                }
                // Check if node is a class to add statements to.
                const clazz = classes.find((cl) => cl.classFunction === node);
                if (clazz) {
                    const functionExpression = node;
                    const newExpressions = clazz.statements.map((st) => ts.createStatement(st.expressionStatement.expression));
                    // Create a new body with all the original statements, plus new ones,
                    // plus return statement.
                    const newBody = ts.createBlock([
                        ...functionExpression.body.statements.slice(0, -1),
                        ...newExpressions,
                        ...functionExpression.body.statements.slice(-1),
                    ]);
                    const newNode = ts.createFunctionExpression(functionExpression.modifiers, functionExpression.asteriskToken, functionExpression.name, functionExpression.typeParameters, functionExpression.parameters, functionExpression.type, newBody);
                    // Replace node with modified one.
                    return ts.visitEachChild(newNode, visitor, context);
                }
                // Otherwise return node as is.
                return ts.visitEachChild(node, visitor, context);
            };
            return ts.visitNode(sf, visitor);
        };
        return transformer;
    };
}
exports.getFoldFileTransformer = getFoldFileTransformer;
function findClassDeclarations(node) {
    const classes = [];
    // Find all class declarations, build a ClassData for each.
    ts.forEachChild(node, (child) => {
        if (child.kind !== ts.SyntaxKind.VariableStatement) {
            return;
        }
        const varStmt = child;
        if (varStmt.declarationList.declarations.length > 1) {
            return;
        }
        const varDecl = varStmt.declarationList.declarations[0];
        if (varDecl.name.kind !== ts.SyntaxKind.Identifier) {
            return;
        }
        const name = varDecl.name.text;
        const expr = varDecl.initializer;
        if (!expr || expr.kind !== ts.SyntaxKind.ParenthesizedExpression) {
            return;
        }
        if (expr.expression.kind !== ts.SyntaxKind.CallExpression) {
            return;
        }
        const callExpr = expr.expression;
        if (callExpr.expression.kind !== ts.SyntaxKind.FunctionExpression) {
            return;
        }
        const fn = callExpr.expression;
        if (fn.body.statements.length < 2) {
            return;
        }
        if (fn.body.statements[0].kind !== ts.SyntaxKind.FunctionDeclaration) {
            return;
        }
        const innerFn = fn.body.statements[0];
        if (fn.body.statements[fn.body.statements.length - 1].kind !== ts.SyntaxKind.ReturnStatement) {
            return;
        }
        if (!innerFn.name || innerFn.name.kind !== ts.SyntaxKind.Identifier) {
            return;
        }
        if (innerFn.name.text !== name) {
            return;
        }
        classes.push({
            name,
            class: varDecl,
            classFunction: fn,
            statements: [],
        });
    });
    return classes;
}
function findClassStaticPropertyAssignments(node, checker, classes) {
    const statements = [];
    // Find each assignment outside of the declaration.
    ts.forEachChild(node, (child) => {
        if (child.kind !== ts.SyntaxKind.ExpressionStatement) {
            return;
        }
        const expressionStatement = child;
        if (expressionStatement.expression.kind !== ts.SyntaxKind.BinaryExpression) {
            return;
        }
        const binEx = expressionStatement.expression;
        if (binEx.left.kind !== ts.SyntaxKind.PropertyAccessExpression) {
            return;
        }
        const propAccess = binEx.left;
        if (propAccess.expression.kind !== ts.SyntaxKind.Identifier) {
            return;
        }
        const symbol = checker.getSymbolAtLocation(propAccess.expression);
        if (!symbol) {
            return;
        }
        const decls = symbol.declarations;
        if (decls == undefined || decls.length === 0) {
            return;
        }
        const hostClass = classes.find((clazz => decls.includes(clazz.class)));
        if (!hostClass) {
            return;
        }
        const statement = { expressionStatement, hostClass };
        hostClass.statements.push(statement);
        statements.push(statement);
    });
    return statements;
}
//# sourceMappingURL=data:application/json;base64,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