"use strict";
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@angular-devkit/core");
const node_1 = require("@angular-devkit/core/node");
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
class ProjectNotFoundException extends core_1.BaseException {
    constructor(projectName) {
        super(`Project '${projectName}' could not be found in Workspace.`);
    }
}
exports.ProjectNotFoundException = ProjectNotFoundException;
class TargetNotFoundException extends core_1.BaseException {
    constructor(projectName, targetName) {
        super(`Target '${targetName}' could not be found in project '${projectName}'.`);
    }
}
exports.TargetNotFoundException = TargetNotFoundException;
class ConfigurationNotFoundException extends core_1.BaseException {
    constructor(projectName, configurationName) {
        super(`Configuration '${configurationName}' could not be found in project '${projectName}'.`);
    }
}
exports.ConfigurationNotFoundException = ConfigurationNotFoundException;
// TODO: break this exception apart into more granular ones.
class BuilderCannotBeResolvedException extends core_1.BaseException {
    constructor(builder) {
        super(`Builder '${builder}' cannot be resolved.`);
    }
}
exports.BuilderCannotBeResolvedException = BuilderCannotBeResolvedException;
class ArchitectNotYetLoadedException extends core_1.BaseException {
    constructor() { super(`Architect needs to be loaded before Architect is used.`); }
}
exports.ArchitectNotYetLoadedException = ArchitectNotYetLoadedException;
class BuilderNotFoundException extends core_1.BaseException {
    constructor(builder) {
        super(`Builder ${builder} could not be found.`);
    }
}
exports.BuilderNotFoundException = BuilderNotFoundException;
class Architect {
    constructor(_workspace) {
        this._workspace = _workspace;
        this._targetsSchemaPath = core_1.join(core_1.normalize(__dirname), 'targets-schema.json');
        this._buildersSchemaPath = core_1.join(core_1.normalize(__dirname), 'builders-schema.json');
        this._architectSchemasLoaded = false;
        this._targetMapMap = new Map();
        this._builderPathsMap = new Map();
        this._builderDescriptionMap = new Map();
        this._builderConstructorMap = new Map();
    }
    loadArchitect() {
        if (this._architectSchemasLoaded) {
            return rxjs_1.of(this);
        }
        else {
            return rxjs_1.forkJoin(this._loadJsonFile(this._targetsSchemaPath), this._loadJsonFile(this._buildersSchemaPath)).pipe(operators_1.concatMap(([targetsSchema, buildersSchema]) => {
                this._targetsSchema = targetsSchema;
                this._buildersSchema = buildersSchema;
                this._architectSchemasLoaded = true;
                // Validate and cache all project target maps.
                return rxjs_1.forkJoin(...this._workspace.listProjectNames().map(projectName => {
                    const unvalidatedTargetMap = this._workspace.getProjectTargets(projectName);
                    return this._workspace.validateAgainstSchema(unvalidatedTargetMap, this._targetsSchema).pipe(operators_1.tap(targetMap => this._targetMapMap.set(projectName, targetMap)));
                }));
            }), operators_1.map(() => this));
        }
    }
    listProjectTargets(projectName) {
        return Object.keys(this._getProjectTargetMap(projectName));
    }
    _getProjectTargetMap(projectName) {
        if (!this._targetMapMap.has(projectName)) {
            throw new ProjectNotFoundException(projectName);
        }
        return this._targetMapMap.get(projectName);
    }
    _getProjectTarget(projectName, targetName) {
        const targetMap = this._getProjectTargetMap(projectName);
        const target = targetMap[targetName];
        if (!target) {
            throw new TargetNotFoundException(projectName, targetName);
        }
        return target;
    }
    getBuilderConfiguration(targetSpec) {
        const { project: projectName, target: targetName, configuration: configurationName, overrides, } = targetSpec;
        const project = this._workspace.getProject(projectName);
        const target = this._getProjectTarget(projectName, targetName);
        const options = target.options;
        let configuration = {};
        if (configurationName) {
            if (!target.configurations) {
                throw new ConfigurationNotFoundException(projectName, configurationName);
            }
            configuration = target.configurations[configurationName];
            if (!configuration) {
                throw new ConfigurationNotFoundException(projectName, configurationName);
            }
        }
        const builderConfiguration = {
            root: project.root,
            sourceRoot: project.sourceRoot,
            projectType: project.projectType,
            builder: target.builder,
            options: Object.assign({}, options, configuration, overrides),
        };
        return builderConfiguration;
    }
    run(builderConfig, partialContext = {}) {
        const context = Object.assign({ logger: new core_1.logging.NullLogger(), architect: this, host: this._workspace.host, workspace: this._workspace }, partialContext);
        let builderDescription;
        return this.getBuilderDescription(builderConfig).pipe(operators_1.tap(description => builderDescription = description), operators_1.concatMap(() => this.validateBuilderOptions(builderConfig, builderDescription)), operators_1.tap(validatedBuilderConfig => builderConfig = validatedBuilderConfig), operators_1.map(() => this.getBuilder(builderDescription, context)), operators_1.concatMap(builder => builder.run(builderConfig)));
    }
    getBuilderDescription(builderConfig) {
        // Check cache for this builder description.
        if (this._builderDescriptionMap.has(builderConfig.builder)) {
            return rxjs_1.of(this._builderDescriptionMap.get(builderConfig.builder));
        }
        return new rxjs_1.Observable((obs) => {
            // TODO: this probably needs to be more like NodeModulesEngineHost.
            const basedir = core_1.getSystemPath(this._workspace.root);
            const [pkg, builderName] = builderConfig.builder.split(':');
            const pkgJsonPath = node_1.resolve(pkg, { basedir, resolvePackageJson: true, checkLocal: true });
            let buildersJsonPath;
            let builderPaths;
            // Read the `builders` entry of package.json.
            return this._loadJsonFile(core_1.normalize(pkgJsonPath)).pipe(operators_1.concatMap((pkgJson) => {
                const pkgJsonBuildersentry = pkgJson['builders'];
                if (!pkgJsonBuildersentry) {
                    return rxjs_1.throwError(new BuilderCannotBeResolvedException(builderConfig.builder));
                }
                buildersJsonPath = core_1.join(core_1.dirname(core_1.normalize(pkgJsonPath)), pkgJsonBuildersentry);
                return this._loadJsonFile(buildersJsonPath);
            }), 
            // Validate builders json.
            operators_1.concatMap((builderPathsMap) => this._workspace.validateAgainstSchema(builderPathsMap, this._buildersSchema)), operators_1.concatMap((builderPathsMap) => {
                builderPaths = builderPathsMap.builders[builderName];
                if (!builderPaths) {
                    return rxjs_1.throwError(new BuilderCannotBeResolvedException(builderConfig.builder));
                }
                // Resolve paths in the builder paths.
                const builderJsonDir = core_1.dirname(buildersJsonPath);
                builderPaths.schema = core_1.join(builderJsonDir, builderPaths.schema);
                builderPaths.class = core_1.join(builderJsonDir, builderPaths.class);
                // Save the builder paths so that we can lazily load the builder.
                this._builderPathsMap.set(builderConfig.builder, builderPaths);
                // Load the schema.
                return this._loadJsonFile(builderPaths.schema);
            }), operators_1.map(builderSchema => {
                const builderDescription = {
                    name: builderConfig.builder,
                    schema: builderSchema,
                    description: builderPaths.description,
                };
                // Save to cache before returning.
                this._builderDescriptionMap.set(builderDescription.name, builderDescription);
                return builderDescription;
            })).subscribe(obs);
        });
    }
    validateBuilderOptions(builderConfig, builderDescription) {
        return this._workspace.validateAgainstSchema(builderConfig.options, builderDescription.schema).pipe(operators_1.map(validatedOptions => {
            builderConfig.options = validatedOptions;
            return builderConfig;
        }));
    }
    getBuilder(builderDescription, context) {
        const name = builderDescription.name;
        let builderConstructor;
        // Check cache for this builder.
        if (this._builderConstructorMap.has(name)) {
            builderConstructor = this._builderConstructorMap.get(name);
        }
        else {
            if (!this._builderPathsMap.has(name)) {
                throw new BuilderNotFoundException(name);
            }
            const builderPaths = this._builderPathsMap.get(name);
            // TODO: support more than the default export, maybe via builder#import-name.
            const builderModule = require(core_1.getSystemPath(builderPaths.class));
            builderConstructor = builderModule['default'];
            // Save builder to cache before returning.
            this._builderConstructorMap.set(builderDescription.name, builderConstructor);
        }
        const builder = new builderConstructor(context);
        return builder;
    }
    _loadJsonFile(path) {
        return this._workspace.host.read(core_1.normalize(path)).pipe(operators_1.map(buffer => core_1.virtualFs.fileBufferToString(buffer)), operators_1.map(str => core_1.parseJson(str, core_1.JsonParseMode.Loose)));
    }
}
exports.Architect = Architect;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXJjaGl0ZWN0LmpzIiwic291cmNlUm9vdCI6Ii4vIiwic291cmNlcyI6WyJwYWNrYWdlcy9hbmd1bGFyX2RldmtpdC9hcmNoaXRlY3Qvc3JjL2FyY2hpdGVjdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7OztHQU1HOztBQUVILCtDQWE4QjtBQUM5QixvREFBbUU7QUFDbkUsK0JBQTREO0FBQzVELDhDQUFxRDtBQUVyRCxNQUFhLHdCQUF5QixTQUFRLG9CQUFhO0lBQ3pELFlBQVksV0FBbUI7UUFDN0IsS0FBSyxDQUFDLFlBQVksV0FBVyxvQ0FBb0MsQ0FBQyxDQUFDO0lBQ3JFLENBQUM7Q0FDRjtBQUpELDREQUlDO0FBRUQsTUFBYSx1QkFBd0IsU0FBUSxvQkFBYTtJQUN4RCxZQUFZLFdBQW1CLEVBQUUsVUFBa0I7UUFDakQsS0FBSyxDQUFDLFdBQVcsVUFBVSxvQ0FBb0MsV0FBVyxJQUFJLENBQUMsQ0FBQztJQUNsRixDQUFDO0NBQ0Y7QUFKRCwwREFJQztBQUVELE1BQWEsOEJBQStCLFNBQVEsb0JBQWE7SUFDL0QsWUFBWSxXQUFtQixFQUFFLGlCQUF5QjtRQUN4RCxLQUFLLENBQUMsa0JBQWtCLGlCQUFpQixvQ0FBb0MsV0FBVyxJQUFJLENBQUMsQ0FBQztJQUNoRyxDQUFDO0NBQ0Y7QUFKRCx3RUFJQztBQUVELDREQUE0RDtBQUM1RCxNQUFhLGdDQUFpQyxTQUFRLG9CQUFhO0lBQ2pFLFlBQVksT0FBZTtRQUN6QixLQUFLLENBQUMsWUFBWSxPQUFPLHVCQUF1QixDQUFDLENBQUM7SUFDcEQsQ0FBQztDQUNGO0FBSkQsNEVBSUM7QUFFRCxNQUFhLDhCQUErQixTQUFRLG9CQUFhO0lBQy9ELGdCQUFnQixLQUFLLENBQUMsd0RBQXdELENBQUMsQ0FBQyxDQUFDLENBQUM7Q0FDbkY7QUFGRCx3RUFFQztBQUVELE1BQWEsd0JBQXlCLFNBQVEsb0JBQWE7SUFDekQsWUFBWSxPQUFlO1FBQ3pCLEtBQUssQ0FBQyxXQUFXLE9BQU8sc0JBQXNCLENBQUMsQ0FBQztJQUNsRCxDQUFDO0NBQ0Y7QUFKRCw0REFJQztBQW9FRCxNQUFhLFNBQVM7SUFXcEIsWUFBb0IsVUFBNEM7UUFBNUMsZUFBVSxHQUFWLFVBQVUsQ0FBa0M7UUFWL0MsdUJBQWtCLEdBQUcsV0FBSSxDQUFDLGdCQUFTLENBQUMsU0FBUyxDQUFDLEVBQUUscUJBQXFCLENBQUMsQ0FBQztRQUN2RSx3QkFBbUIsR0FBRyxXQUFJLENBQUMsZ0JBQVMsQ0FBQyxTQUFTLENBQUMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1FBR2xGLDRCQUF1QixHQUFHLEtBQUssQ0FBQztRQUNoQyxrQkFBYSxHQUFHLElBQUksR0FBRyxFQUFxQixDQUFDO1FBQzdDLHFCQUFnQixHQUFHLElBQUksR0FBRyxFQUF3QixDQUFDO1FBQ25ELDJCQUFzQixHQUFHLElBQUksR0FBRyxFQUE4QixDQUFDO1FBQy9ELDJCQUFzQixHQUFHLElBQUksR0FBRyxFQUFrQyxDQUFDO0lBRVAsQ0FBQztJQUVyRSxhQUFhO1FBQ1gsSUFBSSxJQUFJLENBQUMsdUJBQXVCLEVBQUU7WUFDaEMsT0FBTyxTQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDakI7YUFBTTtZQUNMLE9BQU8sZUFBUSxDQUNiLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEVBQzNDLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLENBQzdDLENBQUMsSUFBSSxDQUNKLHFCQUFTLENBQUMsQ0FBQyxDQUFDLGFBQWEsRUFBRSxjQUFjLENBQUMsRUFBRSxFQUFFO2dCQUM1QyxJQUFJLENBQUMsY0FBYyxHQUFHLGFBQWEsQ0FBQztnQkFDcEMsSUFBSSxDQUFDLGVBQWUsR0FBRyxjQUFjLENBQUM7Z0JBQ3RDLElBQUksQ0FBQyx1QkFBdUIsR0FBRyxJQUFJLENBQUM7Z0JBRXBDLDhDQUE4QztnQkFDOUMsT0FBTyxlQUFRLENBQ2IsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLGdCQUFnQixFQUFFLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxFQUFFO29CQUN0RCxNQUFNLG9CQUFvQixHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsaUJBQWlCLENBQUMsV0FBVyxDQUFDLENBQUM7b0JBRTVFLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxxQkFBcUIsQ0FDMUMsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDLElBQUksQ0FDN0MsZUFBRyxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsV0FBVyxFQUFFLFNBQVMsQ0FBQyxDQUFDLENBQ25FLENBQUM7Z0JBQ0osQ0FBQyxDQUFDLENBQ0gsQ0FBQztZQUNKLENBQUMsQ0FBQyxFQUNGLGVBQUcsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FDaEIsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVELGtCQUFrQixDQUFDLFdBQW1CO1FBQ3BDLE9BQU8sTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRU8sb0JBQW9CLENBQUMsV0FBbUI7UUFDOUMsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxFQUFFO1lBQ3hDLE1BQU0sSUFBSSx3QkFBd0IsQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUNqRDtRQUVELE9BQU8sSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFjLENBQUM7SUFDMUQsQ0FBQztJQUVPLGlCQUFpQixDQUFTLFdBQW1CLEVBQUUsVUFBa0I7UUFDdkUsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBRXpELE1BQU0sTUFBTSxHQUFHLFNBQVMsQ0FBQyxVQUFVLENBQW9CLENBQUM7UUFFeEQsSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUNYLE1BQU0sSUFBSSx1QkFBdUIsQ0FBQyxXQUFXLEVBQUUsVUFBVSxDQUFDLENBQUM7U0FDNUQ7UUFFRCxPQUFPLE1BQU0sQ0FBQztJQUNoQixDQUFDO0lBRUQsdUJBQXVCLENBQVcsVUFBMkI7UUFDM0QsTUFBTSxFQUNKLE9BQU8sRUFBRSxXQUFXLEVBQ3BCLE1BQU0sRUFBRSxVQUFVLEVBQ2xCLGFBQWEsRUFBRSxpQkFBaUIsRUFDaEMsU0FBUyxHQUNWLEdBQUcsVUFBVSxDQUFDO1FBRWYsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDeEQsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFdBQVcsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUMvRCxNQUFNLE9BQU8sR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDO1FBQy9CLElBQUksYUFBYSxHQUF3QixFQUFFLENBQUM7UUFFNUMsSUFBSSxpQkFBaUIsRUFBRTtZQUNyQixJQUFJLENBQUMsTUFBTSxDQUFDLGNBQWMsRUFBRTtnQkFDMUIsTUFBTSxJQUFJLDhCQUE4QixDQUFDLFdBQVcsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO2FBQzFFO1lBRUQsYUFBYSxHQUFHLE1BQU0sQ0FBQyxjQUFjLENBQUMsaUJBQWlCLENBQUMsQ0FBQztZQUV6RCxJQUFJLENBQUMsYUFBYSxFQUFFO2dCQUNsQixNQUFNLElBQUksOEJBQThCLENBQUMsV0FBVyxFQUFFLGlCQUFpQixDQUFDLENBQUM7YUFDMUU7U0FDRjtRQUVELE1BQU0sb0JBQW9CLEdBQW1DO1lBQzNELElBQUksRUFBRSxPQUFPLENBQUMsSUFBWTtZQUMxQixVQUFVLEVBQUUsT0FBTyxDQUFDLFVBQThCO1lBQ2xELFdBQVcsRUFBRSxPQUFPLENBQUMsV0FBVztZQUNoQyxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU87WUFDdkIsT0FBTyxFQUFFLGtCQUNKLE9BQU8sRUFDUCxhQUFhLEVBQ2IsU0FBZSxDQUNQO1NBQ2QsQ0FBQztRQUVGLE9BQU8sb0JBQW9CLENBQUM7SUFDOUIsQ0FBQztJQUVELEdBQUcsQ0FDRCxhQUE2QyxFQUM3QyxpQkFBMEMsRUFBRTtRQUU1QyxNQUFNLE9BQU8sbUJBQ1gsTUFBTSxFQUFFLElBQUksY0FBTyxDQUFDLFVBQVUsRUFBRSxFQUNoQyxTQUFTLEVBQUUsSUFBSSxFQUNmLElBQUksRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksRUFDMUIsU0FBUyxFQUFFLElBQUksQ0FBQyxVQUFVLElBQ3ZCLGNBQWMsQ0FDbEIsQ0FBQztRQUVGLElBQUksa0JBQXNDLENBQUM7UUFFM0MsT0FBTyxJQUFJLENBQUMscUJBQXFCLENBQUMsYUFBYSxDQUFDLENBQUMsSUFBSSxDQUNuRCxlQUFHLENBQUMsV0FBVyxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsR0FBRyxXQUFXLENBQUMsRUFDcEQscUJBQVMsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsc0JBQXNCLENBQUMsYUFBYSxFQUFFLGtCQUFrQixDQUFDLENBQUMsRUFDL0UsZUFBRyxDQUFDLHNCQUFzQixDQUFDLEVBQUUsQ0FBQyxhQUFhLEdBQUcsc0JBQXNCLENBQUMsRUFDckUsZUFBRyxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsa0JBQWtCLEVBQUUsT0FBTyxDQUFDLENBQUMsRUFDdkQscUJBQVMsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FDakQsQ0FBQztJQUNKLENBQUM7SUFFRCxxQkFBcUIsQ0FDbkIsYUFBNkM7UUFFN0MsNENBQTRDO1FBQzVDLElBQUksSUFBSSxDQUFDLHNCQUFzQixDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLEVBQUU7WUFDMUQsT0FBTyxTQUFFLENBQUMsSUFBSSxDQUFDLHNCQUFzQixDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUF1QixDQUFDLENBQUM7U0FDekY7UUFFRCxPQUFPLElBQUksaUJBQVUsQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFO1lBQzVCLG1FQUFtRTtZQUNuRSxNQUFNLE9BQU8sR0FBRyxvQkFBYSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDcEQsTUFBTSxDQUFDLEdBQUcsRUFBRSxXQUFXLENBQUMsR0FBRyxhQUFhLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUM1RCxNQUFNLFdBQVcsR0FBRyxjQUFXLENBQUMsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGtCQUFrQixFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztZQUM5RixJQUFJLGdCQUFzQixDQUFDO1lBQzNCLElBQUksWUFBMEIsQ0FBQztZQUUvQiw2Q0FBNkM7WUFDN0MsT0FBTyxJQUFJLENBQUMsYUFBYSxDQUFDLGdCQUFTLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQ3BELHFCQUFTLENBQUMsQ0FBQyxPQUFtQixFQUFFLEVBQUU7Z0JBQ2hDLE1BQU0sb0JBQW9CLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBVyxDQUFDO2dCQUMzRCxJQUFJLENBQUMsb0JBQW9CLEVBQUU7b0JBQ3pCLE9BQU8saUJBQVUsQ0FBQyxJQUFJLGdDQUFnQyxDQUFDLGFBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO2lCQUNoRjtnQkFFRCxnQkFBZ0IsR0FBRyxXQUFJLENBQUMsY0FBTyxDQUFDLGdCQUFTLENBQUMsV0FBVyxDQUFDLENBQUMsRUFBRSxvQkFBb0IsQ0FBQyxDQUFDO2dCQUUvRSxPQUFPLElBQUksQ0FBQyxhQUFhLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztZQUM5QyxDQUFDLENBQUM7WUFDRiwwQkFBMEI7WUFDMUIscUJBQVMsQ0FBQyxDQUFDLGVBQWUsRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxxQkFBcUIsQ0FDbEUsZUFBZSxFQUFFLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQyxFQUN6QyxxQkFBUyxDQUFDLENBQUMsZUFBZSxFQUFFLEVBQUU7Z0JBQzVCLFlBQVksR0FBRyxlQUFlLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxDQUFDO2dCQUVyRCxJQUFJLENBQUMsWUFBWSxFQUFFO29CQUNqQixPQUFPLGlCQUFVLENBQUMsSUFBSSxnQ0FBZ0MsQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztpQkFDaEY7Z0JBRUQsc0NBQXNDO2dCQUN0QyxNQUFNLGNBQWMsR0FBRyxjQUFPLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztnQkFDakQsWUFBWSxDQUFDLE1BQU0sR0FBRyxXQUFJLENBQUMsY0FBYyxFQUFFLFlBQVksQ0FBQyxNQUFNLENBQUMsQ0FBQztnQkFDaEUsWUFBWSxDQUFDLEtBQUssR0FBRyxXQUFJLENBQUMsY0FBYyxFQUFFLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQztnQkFFOUQsaUVBQWlFO2dCQUNqRSxJQUFJLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxPQUFPLEVBQUUsWUFBWSxDQUFDLENBQUM7Z0JBRS9ELG1CQUFtQjtnQkFDbkIsT0FBTyxJQUFJLENBQUMsYUFBYSxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUNqRCxDQUFDLENBQUMsRUFDRixlQUFHLENBQUMsYUFBYSxDQUFDLEVBQUU7Z0JBQ2xCLE1BQU0sa0JBQWtCLEdBQUc7b0JBQ3pCLElBQUksRUFBRSxhQUFhLENBQUMsT0FBTztvQkFDM0IsTUFBTSxFQUFFLGFBQWE7b0JBQ3JCLFdBQVcsRUFBRSxZQUFZLENBQUMsV0FBVztpQkFDdEMsQ0FBQztnQkFFRixrQ0FBa0M7Z0JBQ2xDLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxHQUFHLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLGtCQUFrQixDQUFDLENBQUM7Z0JBRTdFLE9BQU8sa0JBQWtCLENBQUM7WUFDNUIsQ0FBQyxDQUFDLENBQ0gsQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDbkIsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQsc0JBQXNCLENBQ3BCLGFBQTZDLEVBQUUsa0JBQXNDO1FBRXJGLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxxQkFBcUIsQ0FDMUMsYUFBYSxDQUFDLE9BQU8sRUFBRSxrQkFBa0IsQ0FBQyxNQUFNLENBQ2pELENBQUMsSUFBSSxDQUNKLGVBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFO1lBQ3JCLGFBQWEsQ0FBQyxPQUFPLEdBQUcsZ0JBQWdCLENBQUM7WUFFekMsT0FBTyxhQUFhLENBQUM7UUFDdkIsQ0FBQyxDQUFDLENBQ0gsQ0FBQztJQUNKLENBQUM7SUFFRCxVQUFVLENBQ1Isa0JBQXNDLEVBQUUsT0FBdUI7UUFFL0QsTUFBTSxJQUFJLEdBQUcsa0JBQWtCLENBQUMsSUFBSSxDQUFDO1FBQ3JDLElBQUksa0JBQWdELENBQUM7UUFFckQsZ0NBQWdDO1FBQ2hDLElBQUksSUFBSSxDQUFDLHNCQUFzQixDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUN6QyxrQkFBa0IsR0FBRyxJQUFJLENBQUMsc0JBQXNCLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBaUMsQ0FBQztTQUM1RjthQUFNO1lBQ0wsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUU7Z0JBQ3BDLE1BQU0sSUFBSSx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUMxQztZQUVELE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFpQixDQUFDO1lBRXJFLDZFQUE2RTtZQUM3RSxNQUFNLGFBQWEsR0FBRyxPQUFPLENBQUMsb0JBQWEsQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztZQUNqRSxrQkFBa0IsR0FBRyxhQUFhLENBQUMsU0FBUyxDQUFpQyxDQUFDO1lBRTlFLDBDQUEwQztZQUMxQyxJQUFJLENBQUMsc0JBQXNCLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLElBQUksRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO1NBQzlFO1FBRUQsTUFBTSxPQUFPLEdBQUcsSUFBSSxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUVoRCxPQUFPLE9BQU8sQ0FBQztJQUNqQixDQUFDO0lBRU8sYUFBYSxDQUFDLElBQVU7UUFDOUIsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FDcEQsZUFBRyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsZ0JBQVMsQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUNuRCxlQUFHLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxnQkFBUyxDQUFDLEdBQUcsRUFBRSxvQkFBYSxDQUFDLEtBQUssQ0FBcUIsQ0FBQyxDQUNwRSxDQUFDO0lBQ0osQ0FBQztDQUNGO0FBcFBELDhCQW9QQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQGxpY2Vuc2VcbiAqIENvcHlyaWdodCBHb29nbGUgSW5jLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqIFVzZSBvZiB0aGlzIHNvdXJjZSBjb2RlIGlzIGdvdmVybmVkIGJ5IGFuIE1JVC1zdHlsZSBsaWNlbnNlIHRoYXQgY2FuIGJlXG4gKiBmb3VuZCBpbiB0aGUgTElDRU5TRSBmaWxlIGF0IGh0dHBzOi8vYW5ndWxhci5pby9saWNlbnNlXG4gKi9cblxuaW1wb3J0IHtcbiAgQmFzZUV4Y2VwdGlvbixcbiAgSnNvbk9iamVjdCxcbiAgSnNvblBhcnNlTW9kZSxcbiAgUGF0aCxcbiAgZGlybmFtZSxcbiAgZXhwZXJpbWVudGFsLFxuICBnZXRTeXN0ZW1QYXRoLFxuICBqb2luLFxuICBsb2dnaW5nLFxuICBub3JtYWxpemUsXG4gIHBhcnNlSnNvbixcbiAgdmlydHVhbEZzLFxufSBmcm9tICdAYW5ndWxhci1kZXZraXQvY29yZSc7XG5pbXBvcnQgeyByZXNvbHZlIGFzIG5vZGVSZXNvbHZlIH0gZnJvbSAnQGFuZ3VsYXItZGV2a2l0L2NvcmUvbm9kZSc7XG5pbXBvcnQgeyBPYnNlcnZhYmxlLCBmb3JrSm9pbiwgb2YsIHRocm93RXJyb3IgfSBmcm9tICdyeGpzJztcbmltcG9ydCB7IGNvbmNhdE1hcCwgbWFwLCB0YXAgfSBmcm9tICdyeGpzL29wZXJhdG9ycyc7XG5cbmV4cG9ydCBjbGFzcyBQcm9qZWN0Tm90Rm91bmRFeGNlcHRpb24gZXh0ZW5kcyBCYXNlRXhjZXB0aW9uIHtcbiAgY29uc3RydWN0b3IocHJvamVjdE5hbWU6IHN0cmluZykge1xuICAgIHN1cGVyKGBQcm9qZWN0ICcke3Byb2plY3ROYW1lfScgY291bGQgbm90IGJlIGZvdW5kIGluIFdvcmtzcGFjZS5gKTtcbiAgfVxufVxuXG5leHBvcnQgY2xhc3MgVGFyZ2V0Tm90Rm91bmRFeGNlcHRpb24gZXh0ZW5kcyBCYXNlRXhjZXB0aW9uIHtcbiAgY29uc3RydWN0b3IocHJvamVjdE5hbWU6IHN0cmluZywgdGFyZ2V0TmFtZTogc3RyaW5nKSB7XG4gICAgc3VwZXIoYFRhcmdldCAnJHt0YXJnZXROYW1lfScgY291bGQgbm90IGJlIGZvdW5kIGluIHByb2plY3QgJyR7cHJvamVjdE5hbWV9Jy5gKTtcbiAgfVxufVxuXG5leHBvcnQgY2xhc3MgQ29uZmlndXJhdGlvbk5vdEZvdW5kRXhjZXB0aW9uIGV4dGVuZHMgQmFzZUV4Y2VwdGlvbiB7XG4gIGNvbnN0cnVjdG9yKHByb2plY3ROYW1lOiBzdHJpbmcsIGNvbmZpZ3VyYXRpb25OYW1lOiBzdHJpbmcpIHtcbiAgICBzdXBlcihgQ29uZmlndXJhdGlvbiAnJHtjb25maWd1cmF0aW9uTmFtZX0nIGNvdWxkIG5vdCBiZSBmb3VuZCBpbiBwcm9qZWN0ICcke3Byb2plY3ROYW1lfScuYCk7XG4gIH1cbn1cblxuLy8gVE9ETzogYnJlYWsgdGhpcyBleGNlcHRpb24gYXBhcnQgaW50byBtb3JlIGdyYW51bGFyIG9uZXMuXG5leHBvcnQgY2xhc3MgQnVpbGRlckNhbm5vdEJlUmVzb2x2ZWRFeGNlcHRpb24gZXh0ZW5kcyBCYXNlRXhjZXB0aW9uIHtcbiAgY29uc3RydWN0b3IoYnVpbGRlcjogc3RyaW5nKSB7XG4gICAgc3VwZXIoYEJ1aWxkZXIgJyR7YnVpbGRlcn0nIGNhbm5vdCBiZSByZXNvbHZlZC5gKTtcbiAgfVxufVxuXG5leHBvcnQgY2xhc3MgQXJjaGl0ZWN0Tm90WWV0TG9hZGVkRXhjZXB0aW9uIGV4dGVuZHMgQmFzZUV4Y2VwdGlvbiB7XG4gIGNvbnN0cnVjdG9yKCkgeyBzdXBlcihgQXJjaGl0ZWN0IG5lZWRzIHRvIGJlIGxvYWRlZCBiZWZvcmUgQXJjaGl0ZWN0IGlzIHVzZWQuYCk7IH1cbn1cblxuZXhwb3J0IGNsYXNzIEJ1aWxkZXJOb3RGb3VuZEV4Y2VwdGlvbiBleHRlbmRzIEJhc2VFeGNlcHRpb24ge1xuICBjb25zdHJ1Y3RvcihidWlsZGVyOiBzdHJpbmcpIHtcbiAgICBzdXBlcihgQnVpbGRlciAke2J1aWxkZXJ9IGNvdWxkIG5vdCBiZSBmb3VuZC5gKTtcbiAgfVxufVxuXG5leHBvcnQgaW50ZXJmYWNlIEJ1aWxkZXJDb250ZXh0IHtcbiAgbG9nZ2VyOiBsb2dnaW5nLkxvZ2dlcjtcbiAgaG9zdDogdmlydHVhbEZzLkhvc3Q8e30+O1xuICB3b3Jrc3BhY2U6IGV4cGVyaW1lbnRhbC53b3Jrc3BhY2UuV29ya3NwYWNlO1xuICBhcmNoaXRlY3Q6IEFyY2hpdGVjdDtcbn1cblxuLy8gVE9ETzogdXNlIEJ1aWxkIEV2ZW50IFByb3RvY29sXG4vLyBodHRwczovL2RvY3MuYmF6ZWwuYnVpbGQvdmVyc2lvbnMvbWFzdGVyL2J1aWxkLWV2ZW50LXByb3RvY29sLmh0bWxcbi8vIGh0dHBzOi8vZ2l0aHViLmNvbS9nb29nbGVhcGlzL2dvb2dsZWFwaXMvdHJlZS9tYXN0ZXIvZ29vZ2xlL2RldnRvb2xzL2J1aWxkL3YxXG5leHBvcnQgaW50ZXJmYWNlIEJ1aWxkRXZlbnQge1xuICBzdWNjZXNzOiBib29sZWFuO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEJ1aWxkZXI8T3B0aW9uc1Q+IHtcbiAgcnVuKGJ1aWxkZXJDb25maWc6IEJ1aWxkZXJDb25maWd1cmF0aW9uPFBhcnRpYWw8T3B0aW9uc1Q+Pik6IE9ic2VydmFibGU8QnVpbGRFdmVudD47XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgQnVpbGRlclBhdGhzTWFwIHtcbiAgYnVpbGRlcnM6IHsgW2s6IHN0cmluZ106IEJ1aWxkZXJQYXRocyB9O1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEJ1aWxkZXJQYXRocyB7XG4gIGNsYXNzOiBQYXRoO1xuICBzY2hlbWE6IFBhdGg7XG4gIGRlc2NyaXB0aW9uOiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgQnVpbGRlckRlc2NyaXB0aW9uIHtcbiAgbmFtZTogc3RyaW5nO1xuICBzY2hlbWE6IEpzb25PYmplY3Q7XG4gIGRlc2NyaXB0aW9uOiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgQnVpbGRlckNvbnN0cnVjdG9yPE9wdGlvbnNUPiB7XG4gIG5ldyhjb250ZXh0OiBCdWlsZGVyQ29udGV4dCk6IEJ1aWxkZXI8T3B0aW9uc1Q+O1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIEJ1aWxkZXJDb25maWd1cmF0aW9uPE9wdGlvbnNUID0ge30+IHtcbiAgcm9vdDogUGF0aDtcbiAgc291cmNlUm9vdD86IFBhdGg7XG4gIHByb2plY3RUeXBlOiBzdHJpbmc7XG4gIGJ1aWxkZXI6IHN0cmluZztcbiAgb3B0aW9uczogT3B0aW9uc1Q7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgVGFyZ2V0U3BlY2lmaWVyPE9wdGlvbnNUID0ge30+IHtcbiAgcHJvamVjdDogc3RyaW5nO1xuICB0YXJnZXQ6IHN0cmluZztcbiAgY29uZmlndXJhdGlvbj86IHN0cmluZztcbiAgb3ZlcnJpZGVzPzogUGFydGlhbDxPcHRpb25zVD47XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgVGFyZ2V0TWFwIHtcbiAgW2s6IHN0cmluZ106IFRhcmdldDtcbn1cblxuZXhwb3J0IGRlY2xhcmUgdHlwZSBUYXJnZXRPcHRpb25zPFQgPSBKc29uT2JqZWN0PiA9IFQ7XG5leHBvcnQgZGVjbGFyZSB0eXBlIFRhcmdldENvbmZpZ3VyYXRpb248VCA9IEpzb25PYmplY3Q+ID0gUGFydGlhbDxUPjtcblxuZXhwb3J0IGludGVyZmFjZSBUYXJnZXQ8VCA9IEpzb25PYmplY3Q+IHtcbiAgYnVpbGRlcjogc3RyaW5nO1xuICBvcHRpb25zOiBUYXJnZXRPcHRpb25zPFQ+O1xuICBjb25maWd1cmF0aW9ucz86IHsgW2s6IHN0cmluZ106IFRhcmdldENvbmZpZ3VyYXRpb248VD4gfTtcbn1cblxuZXhwb3J0IGNsYXNzIEFyY2hpdGVjdCB7XG4gIHByaXZhdGUgcmVhZG9ubHkgX3RhcmdldHNTY2hlbWFQYXRoID0gam9pbihub3JtYWxpemUoX19kaXJuYW1lKSwgJ3RhcmdldHMtc2NoZW1hLmpzb24nKTtcbiAgcHJpdmF0ZSByZWFkb25seSBfYnVpbGRlcnNTY2hlbWFQYXRoID0gam9pbihub3JtYWxpemUoX19kaXJuYW1lKSwgJ2J1aWxkZXJzLXNjaGVtYS5qc29uJyk7XG4gIHByaXZhdGUgX3RhcmdldHNTY2hlbWE6IEpzb25PYmplY3Q7XG4gIHByaXZhdGUgX2J1aWxkZXJzU2NoZW1hOiBKc29uT2JqZWN0O1xuICBwcml2YXRlIF9hcmNoaXRlY3RTY2hlbWFzTG9hZGVkID0gZmFsc2U7XG4gIHByaXZhdGUgX3RhcmdldE1hcE1hcCA9IG5ldyBNYXA8c3RyaW5nLCBUYXJnZXRNYXA+KCk7XG4gIHByaXZhdGUgX2J1aWxkZXJQYXRoc01hcCA9IG5ldyBNYXA8c3RyaW5nLCBCdWlsZGVyUGF0aHM+KCk7XG4gIHByaXZhdGUgX2J1aWxkZXJEZXNjcmlwdGlvbk1hcCA9IG5ldyBNYXA8c3RyaW5nLCBCdWlsZGVyRGVzY3JpcHRpb24+KCk7XG4gIHByaXZhdGUgX2J1aWxkZXJDb25zdHJ1Y3Rvck1hcCA9IG5ldyBNYXA8c3RyaW5nLCBCdWlsZGVyQ29uc3RydWN0b3I8e30+PigpO1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgX3dvcmtzcGFjZTogZXhwZXJpbWVudGFsLndvcmtzcGFjZS5Xb3Jrc3BhY2UpIHsgfVxuXG4gIGxvYWRBcmNoaXRlY3QoKSB7XG4gICAgaWYgKHRoaXMuX2FyY2hpdGVjdFNjaGVtYXNMb2FkZWQpIHtcbiAgICAgIHJldHVybiBvZih0aGlzKTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIGZvcmtKb2luKFxuICAgICAgICB0aGlzLl9sb2FkSnNvbkZpbGUodGhpcy5fdGFyZ2V0c1NjaGVtYVBhdGgpLFxuICAgICAgICB0aGlzLl9sb2FkSnNvbkZpbGUodGhpcy5fYnVpbGRlcnNTY2hlbWFQYXRoKSxcbiAgICAgICkucGlwZShcbiAgICAgICAgY29uY2F0TWFwKChbdGFyZ2V0c1NjaGVtYSwgYnVpbGRlcnNTY2hlbWFdKSA9PiB7XG4gICAgICAgICAgdGhpcy5fdGFyZ2V0c1NjaGVtYSA9IHRhcmdldHNTY2hlbWE7XG4gICAgICAgICAgdGhpcy5fYnVpbGRlcnNTY2hlbWEgPSBidWlsZGVyc1NjaGVtYTtcbiAgICAgICAgICB0aGlzLl9hcmNoaXRlY3RTY2hlbWFzTG9hZGVkID0gdHJ1ZTtcblxuICAgICAgICAgIC8vIFZhbGlkYXRlIGFuZCBjYWNoZSBhbGwgcHJvamVjdCB0YXJnZXQgbWFwcy5cbiAgICAgICAgICByZXR1cm4gZm9ya0pvaW4oXG4gICAgICAgICAgICAuLi50aGlzLl93b3Jrc3BhY2UubGlzdFByb2plY3ROYW1lcygpLm1hcChwcm9qZWN0TmFtZSA9PiB7XG4gICAgICAgICAgICAgIGNvbnN0IHVudmFsaWRhdGVkVGFyZ2V0TWFwID0gdGhpcy5fd29ya3NwYWNlLmdldFByb2plY3RUYXJnZXRzKHByb2plY3ROYW1lKTtcblxuICAgICAgICAgICAgICByZXR1cm4gdGhpcy5fd29ya3NwYWNlLnZhbGlkYXRlQWdhaW5zdFNjaGVtYTxUYXJnZXRNYXA+KFxuICAgICAgICAgICAgICAgIHVudmFsaWRhdGVkVGFyZ2V0TWFwLCB0aGlzLl90YXJnZXRzU2NoZW1hKS5waXBlKFxuICAgICAgICAgICAgICAgICAgdGFwKHRhcmdldE1hcCA9PiB0aGlzLl90YXJnZXRNYXBNYXAuc2V0KHByb2plY3ROYW1lLCB0YXJnZXRNYXApKSxcbiAgICAgICAgICAgICAgKTtcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgICk7XG4gICAgICAgIH0pLFxuICAgICAgICBtYXAoKCkgPT4gdGhpcyksXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIGxpc3RQcm9qZWN0VGFyZ2V0cyhwcm9qZWN0TmFtZTogc3RyaW5nKTogc3RyaW5nW10ge1xuICAgIHJldHVybiBPYmplY3Qua2V5cyh0aGlzLl9nZXRQcm9qZWN0VGFyZ2V0TWFwKHByb2plY3ROYW1lKSk7XG4gIH1cblxuICBwcml2YXRlIF9nZXRQcm9qZWN0VGFyZ2V0TWFwKHByb2plY3ROYW1lOiBzdHJpbmcpOiBUYXJnZXRNYXAge1xuICAgIGlmICghdGhpcy5fdGFyZ2V0TWFwTWFwLmhhcyhwcm9qZWN0TmFtZSkpIHtcbiAgICAgIHRocm93IG5ldyBQcm9qZWN0Tm90Rm91bmRFeGNlcHRpb24ocHJvamVjdE5hbWUpO1xuICAgIH1cblxuICAgIHJldHVybiB0aGlzLl90YXJnZXRNYXBNYXAuZ2V0KHByb2plY3ROYW1lKSBhcyBUYXJnZXRNYXA7XG4gIH1cblxuICBwcml2YXRlIF9nZXRQcm9qZWN0VGFyZ2V0PFQgPSB7fT4ocHJvamVjdE5hbWU6IHN0cmluZywgdGFyZ2V0TmFtZTogc3RyaW5nKTogVGFyZ2V0PFQ+IHtcbiAgICBjb25zdCB0YXJnZXRNYXAgPSB0aGlzLl9nZXRQcm9qZWN0VGFyZ2V0TWFwKHByb2plY3ROYW1lKTtcblxuICAgIGNvbnN0IHRhcmdldCA9IHRhcmdldE1hcFt0YXJnZXROYW1lXSBhcyB7fSBhcyBUYXJnZXQ8VD47XG5cbiAgICBpZiAoIXRhcmdldCkge1xuICAgICAgdGhyb3cgbmV3IFRhcmdldE5vdEZvdW5kRXhjZXB0aW9uKHByb2plY3ROYW1lLCB0YXJnZXROYW1lKTtcbiAgICB9XG5cbiAgICByZXR1cm4gdGFyZ2V0O1xuICB9XG5cbiAgZ2V0QnVpbGRlckNvbmZpZ3VyYXRpb248T3B0aW9uc1Q+KHRhcmdldFNwZWM6IFRhcmdldFNwZWNpZmllcik6IEJ1aWxkZXJDb25maWd1cmF0aW9uPE9wdGlvbnNUPiB7XG4gICAgY29uc3Qge1xuICAgICAgcHJvamVjdDogcHJvamVjdE5hbWUsXG4gICAgICB0YXJnZXQ6IHRhcmdldE5hbWUsXG4gICAgICBjb25maWd1cmF0aW9uOiBjb25maWd1cmF0aW9uTmFtZSxcbiAgICAgIG92ZXJyaWRlcyxcbiAgICB9ID0gdGFyZ2V0U3BlYztcblxuICAgIGNvbnN0IHByb2plY3QgPSB0aGlzLl93b3Jrc3BhY2UuZ2V0UHJvamVjdChwcm9qZWN0TmFtZSk7XG4gICAgY29uc3QgdGFyZ2V0ID0gdGhpcy5fZ2V0UHJvamVjdFRhcmdldChwcm9qZWN0TmFtZSwgdGFyZ2V0TmFtZSk7XG4gICAgY29uc3Qgb3B0aW9ucyA9IHRhcmdldC5vcHRpb25zO1xuICAgIGxldCBjb25maWd1cmF0aW9uOiBUYXJnZXRDb25maWd1cmF0aW9uID0ge307XG5cbiAgICBpZiAoY29uZmlndXJhdGlvbk5hbWUpIHtcbiAgICAgIGlmICghdGFyZ2V0LmNvbmZpZ3VyYXRpb25zKSB7XG4gICAgICAgIHRocm93IG5ldyBDb25maWd1cmF0aW9uTm90Rm91bmRFeGNlcHRpb24ocHJvamVjdE5hbWUsIGNvbmZpZ3VyYXRpb25OYW1lKTtcbiAgICAgIH1cblxuICAgICAgY29uZmlndXJhdGlvbiA9IHRhcmdldC5jb25maWd1cmF0aW9uc1tjb25maWd1cmF0aW9uTmFtZV07XG5cbiAgICAgIGlmICghY29uZmlndXJhdGlvbikge1xuICAgICAgICB0aHJvdyBuZXcgQ29uZmlndXJhdGlvbk5vdEZvdW5kRXhjZXB0aW9uKHByb2plY3ROYW1lLCBjb25maWd1cmF0aW9uTmFtZSk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3QgYnVpbGRlckNvbmZpZ3VyYXRpb246IEJ1aWxkZXJDb25maWd1cmF0aW9uPE9wdGlvbnNUPiA9IHtcbiAgICAgIHJvb3Q6IHByb2plY3Qucm9vdCBhcyBQYXRoLFxuICAgICAgc291cmNlUm9vdDogcHJvamVjdC5zb3VyY2VSb290IGFzIFBhdGggfCB1bmRlZmluZWQsXG4gICAgICBwcm9qZWN0VHlwZTogcHJvamVjdC5wcm9qZWN0VHlwZSxcbiAgICAgIGJ1aWxkZXI6IHRhcmdldC5idWlsZGVyLFxuICAgICAgb3B0aW9uczoge1xuICAgICAgICAuLi5vcHRpb25zLFxuICAgICAgICAuLi5jb25maWd1cmF0aW9uLFxuICAgICAgICAuLi5vdmVycmlkZXMgYXMge30sXG4gICAgICB9IGFzIE9wdGlvbnNULFxuICAgIH07XG5cbiAgICByZXR1cm4gYnVpbGRlckNvbmZpZ3VyYXRpb247XG4gIH1cblxuICBydW48T3B0aW9uc1Q+KFxuICAgIGJ1aWxkZXJDb25maWc6IEJ1aWxkZXJDb25maWd1cmF0aW9uPE9wdGlvbnNUPixcbiAgICBwYXJ0aWFsQ29udGV4dDogUGFydGlhbDxCdWlsZGVyQ29udGV4dD4gPSB7fSxcbiAgKTogT2JzZXJ2YWJsZTxCdWlsZEV2ZW50PiB7XG4gICAgY29uc3QgY29udGV4dDogQnVpbGRlckNvbnRleHQgPSB7XG4gICAgICBsb2dnZXI6IG5ldyBsb2dnaW5nLk51bGxMb2dnZXIoKSxcbiAgICAgIGFyY2hpdGVjdDogdGhpcyxcbiAgICAgIGhvc3Q6IHRoaXMuX3dvcmtzcGFjZS5ob3N0LFxuICAgICAgd29ya3NwYWNlOiB0aGlzLl93b3Jrc3BhY2UsXG4gICAgICAuLi5wYXJ0aWFsQ29udGV4dCxcbiAgICB9O1xuXG4gICAgbGV0IGJ1aWxkZXJEZXNjcmlwdGlvbjogQnVpbGRlckRlc2NyaXB0aW9uO1xuXG4gICAgcmV0dXJuIHRoaXMuZ2V0QnVpbGRlckRlc2NyaXB0aW9uKGJ1aWxkZXJDb25maWcpLnBpcGUoXG4gICAgICB0YXAoZGVzY3JpcHRpb24gPT4gYnVpbGRlckRlc2NyaXB0aW9uID0gZGVzY3JpcHRpb24pLFxuICAgICAgY29uY2F0TWFwKCgpID0+IHRoaXMudmFsaWRhdGVCdWlsZGVyT3B0aW9ucyhidWlsZGVyQ29uZmlnLCBidWlsZGVyRGVzY3JpcHRpb24pKSxcbiAgICAgIHRhcCh2YWxpZGF0ZWRCdWlsZGVyQ29uZmlnID0+IGJ1aWxkZXJDb25maWcgPSB2YWxpZGF0ZWRCdWlsZGVyQ29uZmlnKSxcbiAgICAgIG1hcCgoKSA9PiB0aGlzLmdldEJ1aWxkZXIoYnVpbGRlckRlc2NyaXB0aW9uLCBjb250ZXh0KSksXG4gICAgICBjb25jYXRNYXAoYnVpbGRlciA9PiBidWlsZGVyLnJ1bihidWlsZGVyQ29uZmlnKSksXG4gICAgKTtcbiAgfVxuXG4gIGdldEJ1aWxkZXJEZXNjcmlwdGlvbjxPcHRpb25zVD4oXG4gICAgYnVpbGRlckNvbmZpZzogQnVpbGRlckNvbmZpZ3VyYXRpb248T3B0aW9uc1Q+LFxuICApOiBPYnNlcnZhYmxlPEJ1aWxkZXJEZXNjcmlwdGlvbj4ge1xuICAgIC8vIENoZWNrIGNhY2hlIGZvciB0aGlzIGJ1aWxkZXIgZGVzY3JpcHRpb24uXG4gICAgaWYgKHRoaXMuX2J1aWxkZXJEZXNjcmlwdGlvbk1hcC5oYXMoYnVpbGRlckNvbmZpZy5idWlsZGVyKSkge1xuICAgICAgcmV0dXJuIG9mKHRoaXMuX2J1aWxkZXJEZXNjcmlwdGlvbk1hcC5nZXQoYnVpbGRlckNvbmZpZy5idWlsZGVyKSBhcyBCdWlsZGVyRGVzY3JpcHRpb24pO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgT2JzZXJ2YWJsZSgob2JzKSA9PiB7XG4gICAgICAvLyBUT0RPOiB0aGlzIHByb2JhYmx5IG5lZWRzIHRvIGJlIG1vcmUgbGlrZSBOb2RlTW9kdWxlc0VuZ2luZUhvc3QuXG4gICAgICBjb25zdCBiYXNlZGlyID0gZ2V0U3lzdGVtUGF0aCh0aGlzLl93b3Jrc3BhY2Uucm9vdCk7XG4gICAgICBjb25zdCBbcGtnLCBidWlsZGVyTmFtZV0gPSBidWlsZGVyQ29uZmlnLmJ1aWxkZXIuc3BsaXQoJzonKTtcbiAgICAgIGNvbnN0IHBrZ0pzb25QYXRoID0gbm9kZVJlc29sdmUocGtnLCB7IGJhc2VkaXIsIHJlc29sdmVQYWNrYWdlSnNvbjogdHJ1ZSwgY2hlY2tMb2NhbDogdHJ1ZSB9KTtcbiAgICAgIGxldCBidWlsZGVyc0pzb25QYXRoOiBQYXRoO1xuICAgICAgbGV0IGJ1aWxkZXJQYXRoczogQnVpbGRlclBhdGhzO1xuXG4gICAgICAvLyBSZWFkIHRoZSBgYnVpbGRlcnNgIGVudHJ5IG9mIHBhY2thZ2UuanNvbi5cbiAgICAgIHJldHVybiB0aGlzLl9sb2FkSnNvbkZpbGUobm9ybWFsaXplKHBrZ0pzb25QYXRoKSkucGlwZShcbiAgICAgICAgY29uY2F0TWFwKChwa2dKc29uOiBKc29uT2JqZWN0KSA9PiB7XG4gICAgICAgICAgY29uc3QgcGtnSnNvbkJ1aWxkZXJzZW50cnkgPSBwa2dKc29uWydidWlsZGVycyddIGFzIHN0cmluZztcbiAgICAgICAgICBpZiAoIXBrZ0pzb25CdWlsZGVyc2VudHJ5KSB7XG4gICAgICAgICAgICByZXR1cm4gdGhyb3dFcnJvcihuZXcgQnVpbGRlckNhbm5vdEJlUmVzb2x2ZWRFeGNlcHRpb24oYnVpbGRlckNvbmZpZy5idWlsZGVyKSk7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgYnVpbGRlcnNKc29uUGF0aCA9IGpvaW4oZGlybmFtZShub3JtYWxpemUocGtnSnNvblBhdGgpKSwgcGtnSnNvbkJ1aWxkZXJzZW50cnkpO1xuXG4gICAgICAgICAgcmV0dXJuIHRoaXMuX2xvYWRKc29uRmlsZShidWlsZGVyc0pzb25QYXRoKTtcbiAgICAgICAgfSksXG4gICAgICAgIC8vIFZhbGlkYXRlIGJ1aWxkZXJzIGpzb24uXG4gICAgICAgIGNvbmNhdE1hcCgoYnVpbGRlclBhdGhzTWFwKSA9PiB0aGlzLl93b3Jrc3BhY2UudmFsaWRhdGVBZ2FpbnN0U2NoZW1hPEJ1aWxkZXJQYXRoc01hcD4oXG4gICAgICAgICAgYnVpbGRlclBhdGhzTWFwLCB0aGlzLl9idWlsZGVyc1NjaGVtYSkpLFxuICAgICAgICBjb25jYXRNYXAoKGJ1aWxkZXJQYXRoc01hcCkgPT4ge1xuICAgICAgICAgIGJ1aWxkZXJQYXRocyA9IGJ1aWxkZXJQYXRoc01hcC5idWlsZGVyc1tidWlsZGVyTmFtZV07XG5cbiAgICAgICAgICBpZiAoIWJ1aWxkZXJQYXRocykge1xuICAgICAgICAgICAgcmV0dXJuIHRocm93RXJyb3IobmV3IEJ1aWxkZXJDYW5ub3RCZVJlc29sdmVkRXhjZXB0aW9uKGJ1aWxkZXJDb25maWcuYnVpbGRlcikpO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIC8vIFJlc29sdmUgcGF0aHMgaW4gdGhlIGJ1aWxkZXIgcGF0aHMuXG4gICAgICAgICAgY29uc3QgYnVpbGRlckpzb25EaXIgPSBkaXJuYW1lKGJ1aWxkZXJzSnNvblBhdGgpO1xuICAgICAgICAgIGJ1aWxkZXJQYXRocy5zY2hlbWEgPSBqb2luKGJ1aWxkZXJKc29uRGlyLCBidWlsZGVyUGF0aHMuc2NoZW1hKTtcbiAgICAgICAgICBidWlsZGVyUGF0aHMuY2xhc3MgPSBqb2luKGJ1aWxkZXJKc29uRGlyLCBidWlsZGVyUGF0aHMuY2xhc3MpO1xuXG4gICAgICAgICAgLy8gU2F2ZSB0aGUgYnVpbGRlciBwYXRocyBzbyB0aGF0IHdlIGNhbiBsYXppbHkgbG9hZCB0aGUgYnVpbGRlci5cbiAgICAgICAgICB0aGlzLl9idWlsZGVyUGF0aHNNYXAuc2V0KGJ1aWxkZXJDb25maWcuYnVpbGRlciwgYnVpbGRlclBhdGhzKTtcblxuICAgICAgICAgIC8vIExvYWQgdGhlIHNjaGVtYS5cbiAgICAgICAgICByZXR1cm4gdGhpcy5fbG9hZEpzb25GaWxlKGJ1aWxkZXJQYXRocy5zY2hlbWEpO1xuICAgICAgICB9KSxcbiAgICAgICAgbWFwKGJ1aWxkZXJTY2hlbWEgPT4ge1xuICAgICAgICAgIGNvbnN0IGJ1aWxkZXJEZXNjcmlwdGlvbiA9IHtcbiAgICAgICAgICAgIG5hbWU6IGJ1aWxkZXJDb25maWcuYnVpbGRlcixcbiAgICAgICAgICAgIHNjaGVtYTogYnVpbGRlclNjaGVtYSxcbiAgICAgICAgICAgIGRlc2NyaXB0aW9uOiBidWlsZGVyUGF0aHMuZGVzY3JpcHRpb24sXG4gICAgICAgICAgfTtcblxuICAgICAgICAgIC8vIFNhdmUgdG8gY2FjaGUgYmVmb3JlIHJldHVybmluZy5cbiAgICAgICAgICB0aGlzLl9idWlsZGVyRGVzY3JpcHRpb25NYXAuc2V0KGJ1aWxkZXJEZXNjcmlwdGlvbi5uYW1lLCBidWlsZGVyRGVzY3JpcHRpb24pO1xuXG4gICAgICAgICAgcmV0dXJuIGJ1aWxkZXJEZXNjcmlwdGlvbjtcbiAgICAgICAgfSksXG4gICAgICApLnN1YnNjcmliZShvYnMpO1xuICAgIH0pO1xuICB9XG5cbiAgdmFsaWRhdGVCdWlsZGVyT3B0aW9uczxPcHRpb25zVD4oXG4gICAgYnVpbGRlckNvbmZpZzogQnVpbGRlckNvbmZpZ3VyYXRpb248T3B0aW9uc1Q+LCBidWlsZGVyRGVzY3JpcHRpb246IEJ1aWxkZXJEZXNjcmlwdGlvbixcbiAgKTogT2JzZXJ2YWJsZTxCdWlsZGVyQ29uZmlndXJhdGlvbjxPcHRpb25zVD4+IHtcbiAgICByZXR1cm4gdGhpcy5fd29ya3NwYWNlLnZhbGlkYXRlQWdhaW5zdFNjaGVtYTxPcHRpb25zVD4oXG4gICAgICBidWlsZGVyQ29uZmlnLm9wdGlvbnMsIGJ1aWxkZXJEZXNjcmlwdGlvbi5zY2hlbWEsXG4gICAgKS5waXBlKFxuICAgICAgbWFwKHZhbGlkYXRlZE9wdGlvbnMgPT4ge1xuICAgICAgICBidWlsZGVyQ29uZmlnLm9wdGlvbnMgPSB2YWxpZGF0ZWRPcHRpb25zO1xuXG4gICAgICAgIHJldHVybiBidWlsZGVyQ29uZmlnO1xuICAgICAgfSksXG4gICAgKTtcbiAgfVxuXG4gIGdldEJ1aWxkZXI8T3B0aW9uc1Q+KFxuICAgIGJ1aWxkZXJEZXNjcmlwdGlvbjogQnVpbGRlckRlc2NyaXB0aW9uLCBjb250ZXh0OiBCdWlsZGVyQ29udGV4dCxcbiAgKTogQnVpbGRlcjxPcHRpb25zVD4ge1xuICAgIGNvbnN0IG5hbWUgPSBidWlsZGVyRGVzY3JpcHRpb24ubmFtZTtcbiAgICBsZXQgYnVpbGRlckNvbnN0cnVjdG9yOiBCdWlsZGVyQ29uc3RydWN0b3I8T3B0aW9uc1Q+O1xuXG4gICAgLy8gQ2hlY2sgY2FjaGUgZm9yIHRoaXMgYnVpbGRlci5cbiAgICBpZiAodGhpcy5fYnVpbGRlckNvbnN0cnVjdG9yTWFwLmhhcyhuYW1lKSkge1xuICAgICAgYnVpbGRlckNvbnN0cnVjdG9yID0gdGhpcy5fYnVpbGRlckNvbnN0cnVjdG9yTWFwLmdldChuYW1lKSBhcyBCdWlsZGVyQ29uc3RydWN0b3I8T3B0aW9uc1Q+O1xuICAgIH0gZWxzZSB7XG4gICAgICBpZiAoIXRoaXMuX2J1aWxkZXJQYXRoc01hcC5oYXMobmFtZSkpIHtcbiAgICAgICAgdGhyb3cgbmV3IEJ1aWxkZXJOb3RGb3VuZEV4Y2VwdGlvbihuYW1lKTtcbiAgICAgIH1cblxuICAgICAgY29uc3QgYnVpbGRlclBhdGhzID0gdGhpcy5fYnVpbGRlclBhdGhzTWFwLmdldChuYW1lKSBhcyBCdWlsZGVyUGF0aHM7XG5cbiAgICAgIC8vIFRPRE86IHN1cHBvcnQgbW9yZSB0aGFuIHRoZSBkZWZhdWx0IGV4cG9ydCwgbWF5YmUgdmlhIGJ1aWxkZXIjaW1wb3J0LW5hbWUuXG4gICAgICBjb25zdCBidWlsZGVyTW9kdWxlID0gcmVxdWlyZShnZXRTeXN0ZW1QYXRoKGJ1aWxkZXJQYXRocy5jbGFzcykpO1xuICAgICAgYnVpbGRlckNvbnN0cnVjdG9yID0gYnVpbGRlck1vZHVsZVsnZGVmYXVsdCddIGFzIEJ1aWxkZXJDb25zdHJ1Y3RvcjxPcHRpb25zVD47XG5cbiAgICAgIC8vIFNhdmUgYnVpbGRlciB0byBjYWNoZSBiZWZvcmUgcmV0dXJuaW5nLlxuICAgICAgdGhpcy5fYnVpbGRlckNvbnN0cnVjdG9yTWFwLnNldChidWlsZGVyRGVzY3JpcHRpb24ubmFtZSwgYnVpbGRlckNvbnN0cnVjdG9yKTtcbiAgICB9XG5cbiAgICBjb25zdCBidWlsZGVyID0gbmV3IGJ1aWxkZXJDb25zdHJ1Y3Rvcihjb250ZXh0KTtcblxuICAgIHJldHVybiBidWlsZGVyO1xuICB9XG5cbiAgcHJpdmF0ZSBfbG9hZEpzb25GaWxlKHBhdGg6IFBhdGgpOiBPYnNlcnZhYmxlPEpzb25PYmplY3Q+IHtcbiAgICByZXR1cm4gdGhpcy5fd29ya3NwYWNlLmhvc3QucmVhZChub3JtYWxpemUocGF0aCkpLnBpcGUoXG4gICAgICBtYXAoYnVmZmVyID0+IHZpcnR1YWxGcy5maWxlQnVmZmVyVG9TdHJpbmcoYnVmZmVyKSksXG4gICAgICBtYXAoc3RyID0+IHBhcnNlSnNvbihzdHIsIEpzb25QYXJzZU1vZGUuTG9vc2UpIGFzIHt9IGFzIEpzb25PYmplY3QpLFxuICAgICk7XG4gIH1cbn1cbiJdfQ==