<?php
/**
 * Created by PhpStorm.
 * User: amar
 * Date: 2/25/18
 * Time: 12:17 PM
 */

namespace NicoAuth\database\Traits;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use NicoAuth\AuthManager;
use NicoAuth\database\Model\ExplicitPermission;
use NicoAuth\database\Model\PermissionGroup;
use NicoAuth\NicoAuth;


/**
 * Class NicoAuths
 * @package NicoAuth\database\Traits
 */
trait NicoAuths
{
    /**
     * @var \NicoAuth\AuthManager
     */
    protected $authManager;

    /**
     * All of the extended functions
     *
     * @var array
     */
    protected static $extensions = [];

    /**
     * Check if the method call is for role check
     * @param $method
     * @return bool|string
     */
    protected function checkIfMethodIsRole($method)
    {
        $roles = config('nicoAuth.roles', []);

        foreach ($roles as $role => $perm) {

            if ('is' . Str::studly($role) == $method) {
                return $perm;
            }
        }
        return false;

    }

    /**
     * Get associated Groups for this model
     * @return Collection
     * TODO:: This method has to be moved to Auth Manager
     */
    public function getGroups($namespace = null, $namespaceId = null)
    {
        $pgTable = NicoAuth::TABLE_PERMISSION_GROUP;
        $pvTable = NicoAuth::TABLE_USER_PERMISSION_GROUP_PIVOT;
        $query =  PermissionGroup::join($pvTable, "$pvTable.permission_group_id", "$pgTable.perm_grp_id")
            ->where("$pvTable.user_id", $this->getAttribute($this->primaryKey));
        if($namespace) {
            $query->where("{$pgTable}.namespace", $namespace);
        }
        if($namespace && $namespace !='app') {
            $query->where("{$pgTable}.namespace_id", $namespaceId);
        }
        return $query->get();
    }

    /**
     * Get associated explicit permissions for this model
     * @return Collection
     */
    public function getExplicitPermissions()
    {
        return $this->authManager->getExplicitPermissions();
    }

    protected function loadExplicitPermission() {
        return ExplicitPermission::where('user_id', $this->getId())->get();
    }

    /**
     * @param bool $refresh
     * @return AuthManager
     */
    public function loadPermissionAndPermissionGroups($refresh = false, $loadAuthManagerOnly = false)
    {
        if($refresh === true ) {
            $this->authManager = null;
        }

        if ($this->authManager == null) {
            if($loadAuthManagerOnly === true) {
                $this->authManager = new AuthManager($this);
            } else {
                $this->authManager = new AuthManager($this, $this->getGroups(), $this->loadExplicitPermission());
            }
        }
        return $this->authManager;
    }

    public function loadAuthManager () {
        if($this->authManager === null) {
            $this->authManager = new AuthManager($this);
        }
        return $this->authManager;
    }

    public function getAuthManager () {
        return $this->authManager;
    }


    /**
     * Check if the authenticatable is in the given group
     * @param $userGroup
     * @param null $namespaceId
     * @return bool
     */
    public function isA($userGroup, $namespaceId = null)
    {
        return $this->authManager->is($userGroup, $namespaceId);
    }

   public function amI($userGroup)
    {
        return $this->authManager->amI($userGroup);
    }

    public function isRoot () {
        return $this->authManager->isRoot();
    }

    /**
     * @param $permission
     * @param null $namespaceId
     * @return mixed
     */
    public function can($permission, $namespaceId = null)
    {
        return $this->authManager->can($permission, $namespaceId);
    }

    /**
     * Check if authenticatable has group permission.
     * @param $permission
     * @param null $namespaceId
     * @return bool
     */
    public function hasGroupPermission($permission, $namespaceId = null)
    {
        return $this->authManager->hasGroupPermission($permission, $namespaceId);
    }

    /**
     * Check if autheniticatable has explicit permission assigned
     * @param $permission
     * @param null $namespaceId
     * @return bool
     */
    public function hasExplicitPermission($permission, $namespaceId = null)
    {
        return $this->authManager->hasExplicitPermission($permission, $namespaceId);
    }

    /**
     * Assign user to permission group
     * @param PermissionGroup $group
     */
    public function assignGroup(PermissionGroup $group)
    {
        $this->authManager->addToPermissionGroup($group);
    }

    public function unAssignGroup(PermissionGroup $group){
        $this->removeFromGroup($group);
    }

    /**
     * Remove from permission group
     * @param PermissionGroup $group
     */
    public function removeFromGroup(PermissionGroup $group) {
        $this->authManager->removeFromPermissionGroup($group);
    }

    /**
     * @param $permission
     * @param null $namespaceId
     */
    public function permit(array $permission, $namespace = 'app', $namespaceId = null)
    {
        $this->authManager->permit($permission, $namespace, $namespaceId);
    }

    /**
     * @param array $permission
     * @param string $namespace
     * @param null $namespaceId
     */
    public function revoke(array $permission, $namespace = 'app', $namespaceId = null)
    {
        $this->authManager->revoke($permission, $namespace, $namespaceId);
    }

    /**
     * @param array $permission
     * @param string $namespace
     * @param null $namespaceId
     */
    public function rewritePermission(array $permission, $namespace = 'app', $namespaceId = null)
    {
        $this->authManager->permit($permission, $namespace, $namespaceId, ExplicitPermission::MODE_WRITE);
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\BelongsToMany
     */
    public function permissionGroups()
    {
        return $this->belongsToMany(PermissionGroup::class, NicoAuth::TABLE_USER_PERMISSION_GROUP_PIVOT, 'user_id', 'permission_group_id');

    }

    /**
     * Get namespace ids from associated ids.
     * @param $namespace
     * @return array
     */
    public function getNamespaceIds($namespace) {
        $ids = [];
        foreach($this->permissionGroups as $group){
            if($group->namespace == $namespace){
                $ids[] = $group->namespace_id;
            }
        }
        return $ids;
    }

    /**
     * @param $name
     * @param $arguments
     * @return bool
     */
    public function __call($name, $arguments)
    {
        $check = $this->checkIfMethodIsRole($name);
        if ($check !== false) {
            return $this->isA($check, array_get($arguments, 0));
        }
        return parent::__call($name, $arguments);
    }
}
