<?php
/**
 * Created by PhpStorm.
 * User: amar
 * Date: 2/25/18
 * Time: 4:08 PM
 */

namespace NicoAuth;


use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use NicoAuth\Contracts\Authenticatable;
use NicoAuth\database\Model\ExplicitPermission;
use NicoAuth\database\Model\Perm;
use NicoAuth\database\Model\PermissionGroup;
use NicoAuth\Exceptions\InvalidPermissionKeyException;
use NicoAuth\Exceptions\NamespaceIdMissingException;

/**
 * Class AuthManager
 * @package NicoAuth
 */
class AuthManager
{
    /**
     * @var Collection
     */
    protected $groups;
    /**
     * @var PermissionGroup
     */
    protected $newPermissionGroup;

    /**
     * @var Collection
     */
    protected $explicitPermissions;
    /**
     * @var Authenticatable
     */
    protected $authenticatable = null;


    /**
     * AuthManager constructor.
     * @param Authenticatable $model
     * @param array $groups
     * @param array $explicitPermissions
     */
    public function __construct(Authenticatable $model,$groups=[],$explicitPermissions=[])
    {
        $this->authenticatable = $model;
        if(!($groups instanceof Collection)){
            $groups = Collection::make($groups);
        }
        if(!($explicitPermissions instanceof Collection)){
            $explicitPermissions = Collection::make($explicitPermissions);
        }
        $this->explicitPermissions = $explicitPermissions;
        $this->groups = $groups;
    }
    /**
     * Check namespace's validity
     * @param $namespace
     * @param $namespaceId
     * @return bool
     */
    protected function checkNamespaceValidity($namespace,$namespaceId){
        return !($namespace!='app' && $namespaceId==null);
    }

    public function getExplicitPermissions() {
        return $this->explicitPermissions;
    }

    /**
     * @param $namespace
     * @param $namespaceId
     * @return AuthManager
     */
    protected function checkNamespaceValidityOrFail($namespace,$namespaceId){
        if($this->checkNamespaceValidity($namespace,$namespaceId)==false){
            throw new NamespaceIdMissingException();
        }
        return $this;
    }
    /**
     * @param $namespace
     * @param $group
     * @param null $namespaceId
     * @return bool
     */
    protected function checkGroup($namespace, $group,$namespaceId=null){
        $this->checkNamespaceValidityOrFail($namespace,$namespaceId);

        $query =  $this->groups->where('namespace',$namespace)
            ->where('key',$group)
            ->where('user_id',$this->authenticatable->getId())
            ->where('namespace_id',$namespaceId);
        if($query->count()==0){
            return false;
        }

        $group = $query->first();
        $perms = $group->permission;
        if($group->permission=='*'){
            //find all the permission in for the given namespace and id
            $perms = new Permission(Perm::where('namespace',$namespace)->pluck('key')->all());
        }

        foreach ($perms->getValue() as $value){
            if(!$this->hasExplicitGroupPermission($namespace.".{$value}",$namespaceId)){
                return false;
            }
        }
        return true;

    }

    /**
     * Resolve given string into namespace and permission
     * @param string $str
     * @return array An array in format [$namespace,$permission]
     */
    protected function resolveNamespaceAndPermission($str){
        $namespace = substr($str, 0,strpos($str,'.'));
        $permission = substr($str,strpos($str,'.',0)+1);
        return [$namespace,$permission];
    }
    /**
     * @param $group
     * @param int $namespaceId If $group is other than app. The namespaceId must be present
     * @return bool
     */
    public function is($group,$namespaceId=null){
        list($namespace,$permission) = $this->resolveNamespaceAndPermission($group);
        return $this->checkGroup($namespace,$permission,$namespaceId);
    }

        public function amI($userGroup)
    {
        $query =  $this->groups->where('key',$userGroup)
            ->where('user_id',$this->authenticatable->getId());
            // ->where('namespace_id',$namespaceId);
        if($query->count()==0){
            return false;
        }else{
            return true;
        }
    }

    public function isRoot () {
        $rootGroup = $this->groups->filter(function ($group) {
            return $group->namespace === "app" && $group->getOriginalPermissionValue () === "*";
        })->first();
        return !is_null($rootGroup);
    }

    /**
     * Check the group permission for user
     * @param $permission
     * @param null $namespaceId
     * @return bool
     */
    public function hasExplicitGroupPermission($permission,$namespaceId=null){
        return $this->hasExplicitPermission($permission,$namespaceId,'group_permission');
    }

    /**
     * Check if the authenticatable has explicit permissions
     * @param $permission
     * @param null $namespaceId
     * @param string $key
     * @return bool
     */
    public function hasExplicitPermission($permission, $namespaceId=null,$key='explicit_permission'){
        list($namespace,$permission) = $this->resolveNamespaceAndPermission($permission);

        $this-> checkNamespaceValidityOrFail($namespace,$namespaceId);

        $perm = $this->explicitPermissions->where('namespace',$namespace)
            ->where('namespace_id',$namespaceId)->first();

        if($perm){
            return $perm->getAttribute($key)->has($permission);
        }
        return false;
    }

    /**
     * Check if the authenticatable has group permission
     * @param $permission
     * @param null $namespaceId
     * @return bool
     */
    public function hasGroupPermission($permission,$namespaceId = null){



        list($namespace,$permission) = $this->resolveNamespaceAndPermission($permission);

        $this->checkNamespaceValidityOrFail($namespace,$namespaceId);

        foreach ($this->groups as $group){
            //TODO: Possible memory leakage and more CPU requirement.
            //TODO: Implement a better method.
            $group->loadPermissionValues();

            if($group->namespace==$namespace&&$group->namespace_id==$namespaceId && $group->permission->has($permission)){
                return true;
            }
        }
        return false;

    }

    /**
     * @param string|array $permission
     * @param null $namespaceId
     * @return bool
     */
    public function can($permission, $namespaceId=null){

        if($this->hasExplicitPermission($permission,$namespaceId)){
            return true;
        }

        return $this->hasGroupPermission($permission,$namespaceId);

    }

    /**
     * Set authenticatable
     * @param Authenticatable $authenticatable
     */
    public function setAuthenticatable(Authenticatable $authenticatable){
        $this->authenticatable = $authenticatable;
    }

    public function getAuthenticatalble () {
        return $this->authenticatable;
    }

    /**
     * Set permission group
     * @param PermissionGroup $permissionGroup
     * @return AuthManager
     */
    public function addToPermissionGroup(PermissionGroup $permissionGroup){
        if(!$permissionGroup->getAttribute($permissionGroup->getKeyName())){
            $permissionGroup->save();
        }
        $groupId = $permissionGroup->getAttribute($permissionGroup->getKeyName());
        $query = DB::table(NicoAuth::TABLE_USER_PERMISSION_GROUP_PIVOT);
        if($query->where('user_id',$this->authenticatable->getId())
                ->where('permission_group_id',$groupId)->count()==0){
            $query->insert(['user_id'=>$this->authenticatable->getId(),'permission_group_id'=>$groupId]);
        }
        $namespace = $permissionGroup->getAttribute('namespace');
        $namespaceId = $permissionGroup->getAttribute('namespace_id');
        //set group permission
        $perm = ExplicitPermission::where('user_id',$this->authenticatable->getId())
            ->where('namespace',$namespace)
            ->where('namespace_id',$namespaceId)->first();

        if(!$perm){
            $perm = new ExplicitPermission();

            $perm->user_id = $this->authenticatable->getId();
            $perm->namespace = $permissionGroup->namespace;
            $perm->namespace_id = $permissionGroup->namespace_id;
            $perm->explicit_permission = json_encode([]);
            $perm->save();
        }
        $oldPermission = $perm->group_permission;
        $newPermission = $permissionGroup->permission;
        if($newPermission->getValue()=='*'){
            $newPermission = Perm::where('namespace',$namespace)
                ->pluck('key')->toArray();
        }else{
            $newPermission = $newPermission->getValue();
        }

        $oldPermission->add($newPermission);
        $perm->group_permission = $oldPermission;
        $perm->save();
        return $this;
    }

    /**
     * Remove this user from group
     * @param PermissionGroup $permissionGroup
     *
     */
    public function removeFromPermissionGroup (PermissionGroup $permissionGroup) {
        $groupId = $permissionGroup->getAttribute($permissionGroup->getKeyName());
        DB::table(NicoAuth::TABLE_USER_PERMISSION_GROUP_PIVOT)->where('user_id',$this->authenticatable->getId())
            ->where('permission_group_id',$groupId)
            ->delete();
    }

    /**
     * @param array $permissions
     * @param $namespace
     * @param null $namespaceId
     */
    public function validatePermissionOrFail(array $permissions,$namespace,$namespaceId=null){
        $query = Perm::whereIn('key',$permissions)
            ->where('namespace',$namespace);

        $data = $query->pluck('key')->all();

        foreach ($permissions as $permission){
            if(!in_array($permission,$data)){
                throw new InvalidPermissionKeyException("Invalid key " . $permission);
            }
        }
    }

    /**
     * @param array $permissions
     * @param string $namespace
     * @param null|int $namespaceId
     * @param string $mode available options are append|write|remove
     */
    public function permit(array $permissions,$namespace, $namespaceId=null,$mode=ExplicitPermission::MODE_APPEND){
        $this->validatePermissionOrFail($permissions,$namespace,$namespaceId);
        $row = ExplicitPermission::where('namespace',$namespace)
            ->where('namespace_id',$namespaceId)
            ->where('user_id',$this->authenticatable->getId())
            ->first();
        if(!$row){
            $row = new ExplicitPermission();
            $row->forceFill(['namespace'=>$namespace,'namespace_id'=>$namespaceId,'user_id'=>$this->authenticatable->getId(),
                'group_permission'=>json_encode([]),'explicit_permission'=>json_encode([])]);
        }
        $perm = $row->explicit_permission;
        if($mode==ExplicitPermission::MODE_APPEND){
            $perm->add($permissions);
        }elseif($mode==ExplicitPermission::MODE_REMOVE){
            $perm->remove($permissions);
        }elseif($mode==ExplicitPermission::MODE_WRITE){
            $perm = json_encode($permissions);

        }

        $row->explicit_permission = $perm;
        $row->save();

    }

    /**
     * @param array $permissions
     * @param $namespace
     * @param null $namespaceId
     */
    public function revoke(array $permissions,$namespace,$namespaceId=null){
        $this->permit($permissions,$namespace,$namespaceId,ExplicitPermission::MODE_REMOVE);
    }

    /**
     * Create or save a permission group
     * @param $name
     * @param $namespace
     * @param null $namespaceId
     * @return mixed
     */
    public static function createPermissionGroup ($name, $description, $permissions,  $namespace = "app", $namespaceId = null, $id = null) {
        if($namespace!="app" && !$namespaceId) {
            throw new NamespaceIdMissingException();
        }
        if($id) {
            $query = PermissionGroup::where((new PermissionGroup())->getKeyName(),$id);
        }else {
            $query = PermissionGroup::where('namespace',$namespace)
                ->where("key", $name);
        }
        if($namespaceId) {
            $query->where("namespace_id",$namespaceId);
        }

        $group = $query->first();
        if($id && !$group){
            throw new InvalidPermissionKeyException();
        }
        if(!$group) {
            $group = new PermissionGroup();
        }
        $permission = new Permission($permissions);
        $group ->namespace = $namespace; //doesn't matter
        $group ->namespace_id = $namespaceId; // doesn't matter, query was for this particular namespace
        $group -> key = $name;
        $group ->description = $description;
        $group ->permission = $permission;
        $group -> save();

        return $group;

    }

    /**
     * @param $name
     * @param string $namespace
     * @param null $namespaceId
     * @return mixed
     */
    public static function getPermissionGroup ($name,$namespace="app",$namespaceId = null) {
        $query = PermissionGroup::where('namespace', $namespace);
        if(is_numeric($name)) {
           $query->where((new PermissionGroup())->getKeyName(),$name);
        } else {
            $query->where('key', $name);
        }
       if($namespaceId) {
           $query ->where("namespace_id",$namespaceId);
       }
       return $query->first();

    }

    /**
     * @param string $namespace
     * @param null $namespaceId
     * @return Collection
     */
    public static function getAllPermissionGroups ($namespace = "app", $namespaceId = null) {
        $query = PermissionGroup::where('namespace', $namespace)->orderBy('key');
        if($namespaceId) {
            $query ->where("namespace_id",$namespaceId);
        }
        return $query->get();
    }

    /**
     * @param $name
     * @param $namespace
     * @param $namespaceId
     */
    public static function removePermissionGroup ($name, $namespace, $namespaceId) {
        $group = static::getPermissionGroup($name,$namespace,$namespaceId);
        if($group){
            //remove from pivot table first
            DB::table(NicoAuth::TABLE_USER_PERMISSION_GROUP_PIVOT)->where("permission_group_id") ->delete();
            $group->delete ();
        }
    }

    /**
     * @param $namespace
     * @return mixed
     */
    public static function getAvailablePermissions ($namespace) {
        return Perm::where('namespace',$namespace)->get();
    }

    public function removeFromNamespace($namespace = "app", $namespaceId = null) {
        $this->checkNamespaceValidityOrFail($namespace, $namespaceId);
        // get all groups for the namespace
        $query = PermissionGroup::where('namespace',$namespace);
        if($namespaceId != null){
            $query -> where("namespace_id", $namespaceId);
        }
        $groupIds = $query->pluck((new PermissionGroup())->getKeyName())->flatten()->toArray();
        //delete user from the pivot
        DB::table(NicoAuth::TABLE_USER_PERMISSION_GROUP_PIVOT)
            ->whereIn("permission_group_id", $groupIds)->where('user_id', $this->getAuthenticatalble()->getId())->delete();
    }
}
