<?php
/**
 * Created by PhpStorm.
 * User: Amar
 * Date: 11/1/2017
 * Time: 8:51 AM
 */

namespace App\System\Foundation\FileUpload;


use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Intervention\Image\Facades\Image;

trait Upload
{
    /**
     * @param Request $request
     * @param         $storePath
     * @param bool $public
     * @param string $rules
     * @param string $fileKey
     *
     * @return array Array containing image path and thumb path
     * @throws ValidationException
     */
    public function upload(Request $request, $storePath = "", $public = true, $rules = 'required', $fileKey = "file")
    {

        $validator = Validator::make($request->allFiles(), [$fileKey => $rules]);
        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        $file = $request->file($fileKey);
        return $this->uploadFile($file, $storePath, $public);
    }

    /**
     * @param $source
     * @return
     */
    public function moveFromTemp($source)
    {
        $orgSource = $source;

        //if file doesn't content temp dir then return from here
        if(!preg_match('/\/temp\//m',$source))
        {
            return $source;
        }

        $fileSystem = config('filesystems.default', 'local');
        if ($fileSystem == 'gcs') {
            $replace_uri = config('filesystems.disks.gcs.storage_api_uri');
            $source = str_replace($replace_uri . '/', '', $source);
        } else if ($fileSystem == 'local' || $fileSystem == 'public') {
            $source = str_replace(config('app.url'), '', $source);
            $source = str_replace('/storage' . '/', '', $source);
        }

        $destinationPath = str_replace(config('fileupload.temp_dir_folder') . "/", '', $source);

        if (!Storage::exists($source)) {
            return Storage::url($destinationPath);
        }
        if (!Storage::exists($destinationPath)) {
                Storage::move($source, $destinationPath);
        }
        return Storage::url($destinationPath);
    }



    /**
     * core function of file upload
     * @param $file
     * @param string $storePath
     * @param string $public
     * @return array
     */
    public function uploadFile($file, $storePath = '', $public = true)
    {
        $createThumb = false;

        if (!$storePath) {
            switch ($file->getClientMimeType()) {
                case 'image/jpeg':
                case 'image/png':
                case 'image/bmp':
                case 'image/gif':
                    $storePath = config('fileupload.temp_dir_folder') . "/" . config('fileupload.image_directory');
                    $createThumb = config('fileupload.create_thumb');
                    break;
                case 'video/x-flv':
                case 'video/mp4':
                case 'application/x-mpegURL':
                case 'video/MP2T':
                case 'video/3gpp':
                case 'video/quicktime':
                case 'video/x-msvideo':
                case 'video/x-ms-wmv':
                    $storePath = config('fileupload.temp_dir_folder') . "/" . config('fileupload.video_directory');
                    break;
                default:
                    $storePath = config('fileupload.temp_dir_folder') . "/" . config('fileupload.document_directory');
                    break;

            }
        }

        if ($public) {
            $path = $file->storePublicly($storePath);
            $visibility = 'public';
            //remove the public prefix from the path
        } else {
            $path = $file->store($storePath);
            $visibility = 'private';
        }
        $thumbPath = "";

        if ($createThumb === true) {

            try {
                $image = Image::make($file);
                $ratio = $image->height() / $image->width();
                $width = config('fileupload.thumb_width', 100);
                $thumb = $image->resize($width, $ratio * $width);

                $tp =  config('fileupload.image_thumb_directory') . "/" . filename_from_path($path, config('fileupload.thumb_suffix'));

                if (config('filesystems.default', 'local') === 'local') {
                    Storage::put($tp, $image, $visibility);
                    $thumbPath = Storage::url($tp);
                } else {
                    Storage::disk('gcs')->put($tp, $thumb->encode(pathinfo($thumb->basePath(), PATHINFO_EXTENSION), 100));
                    $thumbPath = Storage::disk('gcs')->url($tp);
                }


            } catch (\Exception $exception) {
            }
        }

        return [
            'path' => Storage::url($path),
            'thumb_path' => $thumbPath,
        ];
    }

}
