<?php

namespace App\Modules\UserManagement\Repositories;

use App\Events\User\UserDeleted;
use App\Events\User\UserUpdated;
use App\Modules\UserManagement\Exceptions\ResellerAdminDeletionException;
use App\Modules\UserManagement\Interfaces\UserManagementInterface;
use App\Modules\UserManagement\Database\Models\User;
use App\System\AppConstants;
use App\System\Foundation\Repositories\AppBaseRepository;
use App\System\Scopes\GlobalQueryScope;
use App\System\Site\Database\Models\Reseller;
use App\System\Site\Database\Models\Site;
use App\System\User\Filter\UsersFilter;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use NicoSystem\Exceptions\NicoBadRequestException;

/**
 * Class UserManagementRepository
 * @package App\Modules\UserManagements\Repositories
 */
class UserManagementRepository extends AppBaseRepository implements UserManagementInterface
{
    /**
     * UserManagementRepository constructor.
     * @param User $model
     */
    public function __construct(User $model)
    {
        parent::__construct($model);

    }

    /**
     * @param Builder $builder
     * @return UsersFilter|null
     */
    public function getFilter(Builder $builder)
    {
        return new UsersFilter($builder);
    }

    /**
     * @param array $inputs
     * @return mixed
     */
    public function create(array $inputs)
    {
        $plainPassword = $inputs['plain_password'];
        unset($inputs['plain_password']);
        $user = $this->getUserByEmail($inputs['email']);
        if ($user) {
            throw new NicoBadRequestException("User with email " . $user->email . " already exists.", AppConstants::ERR_DUPLICATE_RESOURCE);
        }

        $user = $this->getUserByEmail($inputs['email'], true);

        if (!$user) {
            $user = parent::create($inputs);
            event(new UserUpdated($user, $plainPassword, true));
            $newUser = true;
        } else {
            event(new UserUpdated($user, null, false));
            $newUser = false;
        }
        $user->new_user = $newUser;
        return $user;
    }

    /**
     * @param $id
     * @param array $attributes
     * @return
     */
    public function getById($id, array $attributes = [])
    {
        return $this->getQuery()->findOrFail($id);
    }

    /**
     * @param $id
     * @param $status
     * @return mixed
     */
    public function changeStatus($id, $status)
    {
        return $this->updateSingle($id, 'status', $status);
    }

    public function destroy($id, array $options = [])
    {
        if (Auth::user()->id == $id) {
            // user cannot remove himself
            throw new NicoBadRequestException("User cannot delete himself.", AppConstants::ERR_NOT_SUFFICIENT_PERMISSION);
        }

        //Deletion of user will be very different. Therefore, we'll only fire the event.
        event(new UserDeleted($this->model->findOrFail($id), AppConstants::getInstance()->getReseller(), "reseller"));
    }

    /**
     * @param $email
     * @return mixed
     */
    private function getUserByEmail($email, $globalQuery = false)

    {
        $query = $this->model;
        if ($globalQuery === true) {
            $query = $query->newQueryWithoutScope(GlobalQueryScope::class);
        }
        return $query->where('email', $email)->first();
    }

}
