<?php

namespace App\Modules\UserManagement\Controllers;

use App\Events\User\BeforeUserResponse;
use App\Modules\UserManagement\Database\Models\User;
use App\Modules\UserManagement\Interfaces\UserManagementInterface;
use App\Modules\UserManagement\Mails\UserInvitation;
use App\Modules\UserManagement\Requests\UserManagementCreateRequest;
use App\System\Foundation\FileUpload\Upload;    
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use NicoSystem\Controllers\BaseController;
use NicoSystem\Data\Status;
use NicoSystem\Requests\NicoRequest;

/**
 * Class UserManagementsController
 * @package App\Modules\UserManagements\Controllers
 */
class UserManagementsController extends BaseController
{
    use Upload;

    /**
     * UserManagementsController constructor.
     * @param UserManagementInterface $repository
     */
    public function __construct(UserManagementInterface $repository)
    {
        $this->repository = $repository;
    }

    /**
     * @param NicoRequest $request
     * @return \Illuminate\Http\Response
     */
    public function index(NicoRequest $request)
    {
        return $this->responseOk($this->repository->getList($request->all()));
    }

    /**
     * @param UserManagementCreateRequest $request
     * @return \Illuminate\Http\Response
     * @throws ValidationException
     */
    public function store(UserManagementCreateRequest $request)
    {
        $data = $request->except('avatar');
        if ($request->hasFile('avatar')) {
            $image = $this->upload($request, '', true, 'required', 'avatar');
            $data['image_url'] = $image['path'];
        }

        if (!$request->has('username'))
            $data['username'] = $data['email'];

        $password = str_random(10);
        $data['password'] = bcrypt($password);
        $data['plain_password'] = $password;
        $data['is_locked'] = true;
        $data['status'] = Status::STATUS_PUBLISHED;
        $user = $this->repository->create($data);

        return $this->responseOk($user);
    }

    /**
     * @param NicoRequest $request
     * @param $id
     * @return \Illuminate\Http\Response|mixed
     */
    public function show(NicoRequest $request, $id)
    {
        $user = $this->repository->getById($id);
        event(new BeforeUserResponse($user));
        return $this->responseOk($user);
    }

    /**
     * @param $id
     * @return mixed
     */
    public function destroy($id)
    {
        $this->repository->destroy($id);
        return $this->responseOk('');
    }


    /**
     * @param NicoRequest $request
     * @param $id
     * @return mixed
     * @throws ValidationException
     */
    public function changeStatus(NicoRequest $request, $id)
    {
        $validator = Validator::make($request->all(), ['status' => 'required']);

        if ($validator->fails()) {
            throw new ValidationException($validator);
        }
        return $this->responseOk($this->repository->changeStatus($id, $request->post('status')), 'Status changed Successfully');
    }

    /**
     * @param User $user
     * @param $password
     */
    public function sendEmail(User $user, $password)
    {
        Mail::to($user->email)->queue(new UserInvitation($user, $password));
    }

    /**
     * @param $id
     * @return mixed
     */
    public function suspend($id)
    {
        return $this->responseOk($this->repository->changeStatus($id, User::STATUS_INACTIVE), 'User Suspended Successfully');
    }

}
