<?php

namespace App\Modules\IsbnApplications\Repositories;

use App\Events\IsbnApplication\IsbnIssuedCommitted;
use App\Exceptions\ActivePublisherPrefixMissingException;
use App\Modules\IsbnApplications\Database\Models\IsbnApplication;
use App\Modules\IsbnApplications\Database\Models\IssuedIsbn;
use App\Modules\IsbnApplications\Exceptions\IsbnQuoataNotAvailableException;
use App\Modules\IsbnApplications\Exceptions\MultipleIsbnIssueAttemptException;
use App\Modules\IsbnApplications\Filters\IsbnFilter;
use App\Modules\IsbnApplications\Interfaces\IssuedIsbnInterface;
use App\System\AppConstants;
use App\System\Publisher\Database\Models\Publisher;
use Google\Cloud\Core\Exception\BadRequestException;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Event;
use NicoSystem\Exceptions\NicoBadRequestException;
use NicoSystem\Exceptions\NicoException;
use NicoSystem\Exceptions\ResourceInUseException;
use NicoSystem\Foundation\Database\BaseModel;
use NicoSystem\Repositories\BaseRepository;

/**
 * Class IsbnApplicationRepository
 * @package App\Modules\IsbnApplications\Repositories
 */
class IssuedIsbnRepository extends BaseRepository implements IssuedIsbnInterface
{
    protected $isbnApplication;
    protected $publisher;

    /**
     * IssuedIsbnRepository constructor.
     * @param IssuedIsbn $model
     * @param IsbnApplication $isbnApplication
     * @param Publisher $publisher
     */
    public function __construct(IssuedIsbn $model, IsbnApplication $isbnApplication, Publisher $publisher)
    {
        parent::__construct($model);
        $this->isbnApplication = $isbnApplication;
        $this->publisher = $publisher;
    }
    // This method is called in every action just in case we've got request for isbns issued for a given application. If not, all isbns will be returned
    protected function attachApplicationIdFilterToMainQueryForAttribute(array $attributes = []) {
        if(array_get($attributes,'application_id')) {
            $this->getQuery()->where('isbn_application_id', array_get($attributes, 'application_id'));
        }
    }

    public function getFilter(Builder $builder)
    {
        return new IsbnFilter($builder);
    }


    public function getById($id, array $attributes = [])
    {
        $this->attachApplicationIdFilterToMainQueryForAttribute($attributes);
        return parent::getById($id, $attributes); 
    }

    protected function save(array $attributes, $id = null)
    {
        if($id) {
            $model = $this->getQuery()->findOrFail($id);
            if ($model -> readonly == 1) {
                throw new NicoBadRequestException("Resource is not editable", AppConstants::ERR_MODEL_NOT_EDITABLE);
            }
        }
        return parent::save($attributes, $id);
    }

    public function create(array $inputs)
    {
        $this->attachApplicationIdFilterToMainQueryForAttribute($inputs);
        $application = $this->isbnApplication -> find (array_get($inputs, 'application_id'));
        if($application -> issuedIsbn () -> first () ) {
            throw new MultipleIsbnIssueAttemptException();
        }
        return parent::create($inputs); 
    }

    public function getList(array $params = [], $paginate = true, array $attributes = [])
    {
        // For Issued ISBN, we'll be returning ISBN that has isbns issued.
        $isbnTable = $this->model->getTable();
        $appTable = $this->isbnApplication->getTable();
        $isbnQuery = $this->isbnApplication -> newQuery();
        $isbnQuery -> leftJoin($isbnTable, function (JoinClause $join) use ($isbnTable, $appTable) {
            $join->on( "{$isbnTable}.isbn_application_id", '=', "{$appTable}.id")
                ->whereNull("{$isbnTable}.deleted_at");

        })
            ->select("{$appTable}.*")
            ->whereNotNull("{$isbnTable}.id")
            ->with(['publisher' => function ($query) {
                $query->select ('id','title');
            }, 'issuedIsbn' => function ($query){
                $query->select('id','publisher_id','isbn_application_id','publisher_prefix_id','isbn_number','issued_date','readonly','full_isbn_number');
            },'authors']);

        $this->query = $isbnQuery;

         return parent::getList($params, $paginate, $attributes);
    }

    public function destroy($id, array $options = [])
    {
        $this->attachApplicationIdFilterToMainQueryForAttribute($options);
        $model = $this->getQuery()->findOrFail($id);
        if($model->readonly) {
            throw new NicoBadRequestException("The committed ISBN cannot be deleted.", AppConstants::ERR_MODEL_NOT_EDITABLE);

        }
        return parent::destroy($id, $options); 
    }

    public function update($id, array $attributes)
    {
        $this->attachApplicationIdFilterToMainQueryForAttribute($attributes);
        return parent::update($id, $attributes); 
    }

    protected function beforeSavingModel(BaseModel $model, array $attributes = [])
    {

        $application = $this->isbnApplication->where('id',array_get($attributes, 'application_id')) -> select('publisher_id') ->first ();
        $query = $this->model->where('isbn_number', $model->isbn_number);
        if(!$application) {
            throw new NicoBadRequestException();
        }
        $publisher = $this->publisher->find ($application->publisher_id);

        if(!$model->id) {
            $activePrefix = $publisher->getActivePrefix();

        } else {
            $activePrefix = $model->publisherPrefix;
            $query->where('id','<>', $model->id);
        }
        if(!$activePrefix) {
            throw new ActivePublisherPrefixMissingException();
        }
        $duplicate = $query ->where('publisher_prefix_id', $activePrefix->id)-> count ();

        if($duplicate > 0) {
            throw new NicoBadRequestException("ISBN already issued.", 'err_duplicate_isbn_number_issue');
        }
        if(!$model->id) {
            // we want ot save set these value only once
            $model -> setAttribute('isbn_application_id', array_get($attributes,'application_id'));
            $model -> setAttribute('publisher_prefix_id', $activePrefix->id);
            $model -> setAttribute('publisher_id',$application->publisher_id);
	}

	$isbnNumber = get_isbn_number(config('isbn.country_prefix','978-9937'),$activePrefix->prefix, $model->isbn_number);
        $model -> setAttribute("full_isbn_number", $isbnNumber.get_isbn_mdd_13_digit($isbnNumber));
        

        parent::beforeSavingModel($model); 
    }

    public function commit($applicationId, $id) {
        $application = $this->isbnApplication->findOrFail($applicationId);
        $isbn =$application ->issuedIsbn()->findOrFail($id);
        $isbn -> readonly = 1;
        $isbn->save ();
        Event::dispatch(new IsbnIssuedCommitted($isbn));
        return $isbn;
    }

    public function generateNextIsbn($applicationId) {
        $app = $this->isbnApplication->findOrFail($applicationId);
        $publisher = $app->publisher;
        if(!$publisher) {
            throw new NicoException("That shouldn't have happened.","err_app_publisher_missing");
        }

        $activePrefix = $publisher->getActivePrefix ();
        if(!$activePrefix) {
            throw new ActivePublisherPrefixMissingException();
        }
        if($activePrefix -> prefix == 0 || $activePrefix -> prefix == 2) {
            $oldIsbn = $this->model -> orderBy ('isbn_number', 'desc') -> first ();
        } else {
            $oldIsbn = $this->model->where('publisher_id', $publisher->id)
                ->where('publisher_prefix_id', $activePrefix->id)
                ->orderBy('isbn_number', 'desc')
                ->first();
        }

        $newIsbn = $this->model->newInstance();
        $prefixDigitCount = strlen($activePrefix->prefix);
        if($oldIsbn) {
            $newIsbn->isbn_number = $oldIsbn->isbn_number + 1;
            // 5 is the allocated space for isbn
//            if ((strlen($newIsbn->isbn_number) + strlen($prefixDigitCount)) > 5  ) {
//               throw new IsbnQuoataNotAvailableException();
//            }
        } else {
            $newIsbn->isbn_number = 0;
        }
        return $newIsbn;
    }
}
