<?php

namespace App\Modules\IsbnApplications\Repositories;

use App\Events\IsbnApplication\Created;
use App\Events\IsbnApplication\Updated;
use App\Modules\IsbnApplications\Filters\IsbnApplicationsFilter;
use App\Modules\IsbnApplications\Interfaces\IsbnApplicationInterface;
use App\Modules\IsbnApplications\Database\Models\IsbnApplication;
use App\System\AppConstants;
use App\System\IsbnApplication\Database\Models\Author;
use App\System\IsbnApplication\Database\Models\IssuedIsbn;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Query\JoinClause;
use NicoSystem\Exceptions\NicoBadRequestException;
use NicoSystem\Filters\BaseFilter;
use NicoSystem\Repositories\BaseRepository;

/**
 * Class IsbnApplicationRepository
 * @package App\Modules\IsbnApplications\Repositories
 */
class IsbnApplicationRepository extends BaseRepository implements IsbnApplicationInterface
{
    protected $issueIsbn;

    protected $author;

    /**
     * IsbnApplicationRepository constructor.
     * @param IsbnApplication $model
     * @param IssuedIsbn $isbn
     * @param Author $author
     */
    public function __construct(IsbnApplication $model, IssuedIsbn $isbn, Author $author)
    {
        parent::__construct($model);
        $this->events['created'] = Created::class;
        $this->events['updated'] = Updated::class;
        $this->issueIsbn = $isbn;
        $this->author = $author;
    }

    /**
     * @param Builder $builder
     * @return BaseFilter
     */
    public function getFilter(Builder $builder)
    {
        return new IsbnApplicationsFilter($builder);
    }


    public function onBeforeResult(Builder $builder)
    {
        $builder->with('publisher');
        parent::onBeforeResult($builder);
    }

    public function getById($id, array $attributes = [])
    {
        return $this->getQuery() -> with('publisher', 'authors')->findOrFail($id);
    }

    public function getList(array $params = [], $paginate = true, array $attributes = [])
    {
        // For Issued ISBN, we'll be returning ISBN that has isbns issued.
        $appTable = $this->model->getTable();
        $isnTbl = $this->issueIsbn->getTable();
        $isbnQuery = $this->model -> newQuery();
        $isbnQuery ->  leftJoin($isnTbl, function (JoinClause $join) use ($isnTbl, $appTable) {
            $join->on( "{$isnTbl}.isbn_application_id", '=', "{$appTable}.id")
                ->whereNull("{$isnTbl}.deleted_at");

        })
            ->select("{$appTable}.*")
            ->with(['publisher' => function ($query) {
            }, 'issuedIsbn' => function ($query){
                $query->select('id','publisher_id','isbn_application_id','publisher_prefix_id','isbn_number','issued_date','readonly','full_isbn_number');
            },'authors']);

        $this->query = $isbnQuery;
        return parent::getList($params, $paginate, $attributes);
    }

    protected function save(array $attributes, $id = null)
    {

        if($id) {
            $model = $this->getById($id);
            if ($model -> readonly == 1) {
                throw new NicoBadRequestException("Resource is not editable", AppConstants::ERR_MODEL_NOT_EDITABLE);
            }
        } else {
            $attributes['requested_date'] = new Carbon();
        }

        if ($id) {
            $model = $this->getQuery()->findOrFail($id);
        } else {
            $model = $this->model->newInstance();
        }
        $model->fill($attributes);
        $this->beforeSavingModel($model, $attributes);
        $this->dispatchCreatingUpdatingEvent($id!=null, $model);
        $model->save();

        $authors = array_get($attributes, 'authors');

        if ($authors && is_array($authors) && count($authors)) {
            // delete all previous records
            $model->authors() -> delete ();
            foreach ($authors as $author) {
                $item = $this->author->newInstance();
                $item -> name = array_get($author, 'name');
                $model->authors() -> save($item);
            }
        }

        $this->dispatchCreatedUpdatedEvent($id !=null, $model);
        return $model;
    }

    public function commitChangesPermanently($id) {
        $model = $this->getById($id);
        $model -> readonly = 1;
        $model ->save ();
    }

    public function markAsReceived ($id) {
        $model = $this->getById($id);
        $model -> received_status = 1;
        $model ->save ();
    }

}
