<?php

namespace App\Modules\IsbnApplications\Repositories;

use App\Events\IsbnApplication\Created;
use App\Events\IsbnApplication\Updated;
use App\Modules\IsbnApplications\Filters\IsbnApplicationsFilter;
use App\Modules\IsbnApplications\Interfaces\IsbnApplicationInterface;
use App\Modules\IsbnApplications\Database\Models\IsbnApplication;
use App\System\AppConstants;
use App\System\IsbnApplication\Database\Models\Author;
use App\System\IsbnApplication\Database\Models\ApplicationIllustration;
use App\System\IsbnApplication\Database\Models\IssuedIsbn;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Query\JoinClause;
use NicoSystem\Exceptions\NicoBadRequestException;
use NicoSystem\Filters\BaseFilter;
use NicoSystem\Repositories\BaseRepository;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;

/**
 * Class IsbnApplicationRepository
 * @package App\Modules\IsbnApplications\Repositories
 */
class IsbnApplicationRepository extends BaseRepository implements IsbnApplicationInterface
{
    protected $issueIsbn;

    protected $author;

    protected $applicationIllustration;

    /**
     * IsbnApplicationRepository constructor.
     * @param IsbnApplication $model
     * @param IssuedIsbn $isbn
     * @param Author $author
     */
    public function __construct(IsbnApplication $model, IssuedIsbn $isbn, Author $author, ApplicationIllustration $applicationIllustration)
    {
        parent::__construct($model);
        $this->events['created'] = Created::class;
        $this->events['updated'] = Updated::class;
        $this->issueIsbn = $isbn;
        $this->author = $author;
        $this->applicationIllustration =  $applicationIllustration;

    }

    /**
     * @param Builder $builder
     * @return BaseFilter
     */
    public function getFilter(Builder $builder)
    {
        return new IsbnApplicationsFilter($builder);
    }


    public function onBeforeResult(Builder $builder)
    {
        $builder->with('publisher');
        parent::onBeforeResult($builder);
    }

    public function getById($id, array $attributes = [])
    {
        return $this->getQuery()->with('publisher', 'authors', 'illustrations')->findOrFail($id);
    }

    public function getLatestId()
    {

        $lastData = DB::table('isbn_applications')->latest('id')->first();
        
        return $lastData->id + 1 ;
        
    }

    public function getList(array $params = [], $paginate = true, array $attributes = [])
    {
        // For Issued ISBN, we'll be returning ISBN that has isbns issued.
        $appTable = $this->model->getTable();
        $isnTbl = $this->issueIsbn->getTable();
        $isbnQuery = $this->model -> newQuery();
        $isbnQuery ->  leftJoin($isnTbl, function (JoinClause $join) use ($isnTbl, $appTable) {
            $join->on( "{$isnTbl}.isbn_application_id", '=', "{$appTable}.id")
                ->whereNull("{$isnTbl}.deleted_at");

        })
	->select("{$appTable}.*");
       	if(isset($params['isbnStatus']) && $params['isbnStatus'] == 'issued') {
                $isbnQuery ->whereNotNull("{$isnTbl}.id");
        } elseif (isset($params['isbnStatus']) && $params['isbnStatus'] == 'non_issued') {
               $isbnQuery->whereNull("{$isnTbl}.id");
        }
           $isbnQuery ->with(['publisher' => function ($query) {
            }, 'issuedIsbn' => function ($query){
                $query->select('id','publisher_id','isbn_application_id','publisher_prefix_id','isbn_number','issued_date','readonly','full_isbn_number');
            },'authors','illustrations']);

        $this->query = $isbnQuery;
        return parent::getList($params, $paginate, $attributes);
    }

    protected function save(array $attributes, $id = null)
    {
        if($id) {
            $model = $this->getById($id);
            // clinet wants to edit the applicaiton event it is issued or commited.
            // if ($model -> readonly == 1) {
            //     throw new NicoBadRequestException("Resource is not editable", AppConstants::ERR_MODEL_NOT_EDITABLE);
            // }
        }
//	 $req_date = new Carbon($attributes['requested_date']);
      //     $attributes['requested_date'] = $req_date->addDay();
//	$attributes['requested_date'] = $req_date->format('Y-m-d');
        if ($id) {
		$model = $this->getQuery()->findOrFail($id);
		$modelId = null;
        } else {
            $model = $this->model->newInstance();

           $model->id = $modelId = (!empty($attributes['application_id'])) ? $attributes['application_id'] : $this->getLatestId();            
        }
        $model->fill($attributes); 
	$this->beforeSavingModel($model, $attributes);
        $this->dispatchCreatingUpdatingEvent($id!=null, $model);
		$model->save();
	$authors = array_get($attributes, 'authors');
        if ($authors && is_array($authors) && count($authors)) {
            // delete all previous records
            $model->authors()->delete();
            foreach ($authors as $author) {
        //      $item = $this->author->newInstance();
	//	$item->name = array_get($author, 'name');
	//	$model->authors()->save($item);
	    
		$auth['name'] = array_get($author, 'name');
                $auth['isbn_application_id'] = ($id) ? $id : $modelId;
                Author::create($auth);
            }
	}
        $illustrationtype = array_get($attributes, 'illustration_type');
        if ($illustrationtype && is_array($illustrationtype) && count($illustrationtype)) {
            // delete all previous records
            $model->illustrations()->delete();
            foreach ($illustrationtype as $illustration) {
                //$item = $this->applicationIllustration->newInstance();
                //$item->name = array_get($illustration, 'item_text');
                //$item->illustration_number = array_get($illustration, 'item_id');
		//$model->illustrations()->save($item);

		$illu['name'] = array_get($illustration, 'item_text');
                $illu['illustration_number'] = array_get($illustration, 'item_id');
                $illu['isbn_application_id'] = ($id) ? $id : $modelId;
		ApplicationIllustration::create($illu);

            }
        }
       
        $this->dispatchCreatedUpdatedEvent($id !=null, $model);
        return $model;
    }

    public function commitChangesPermanently($id) {
        $model = $this->getById($id);
        $model -> readonly = 1;
        $model ->save ();
    }

    public function markAsReceived ($id, $data) {
        $model = $this->getById($id);
	$model -> received_status = ($data['value'] == 1) ? 1 : 0;
	$model ->save ();
	if($data['value'] == 1) {
        	$receiver = $model->publisher->email;
        	$data = [
            		'receiver'=>$model->publisher->title,
            		'title' => "Your ISBN application has been marked as received",
            		'application' => $model,
        	];
        	Mail::send('emails.email-mark-as-received', $data, function($message) use ($model) {
           		 $message->to($model->publisher->email, "ISBN")
                    		->subject('Your ISBN application has been marked as received');
		});
	}
    }


    public function destroy($id, array $options = []) {
        $application = $this->getById($id);

        $isbn =$application->issuedIsbn()->first();
      
        // $this->dispatchEvent('deleting', $model);
        $application->delete();
        // $this->dispatchEvent('deleted', $model);
        return true;
    }
}
