<?php

namespace App\Http\Middleware;

use App\Exceptions\ForbiddenException;
use App\System\AppConstants;
use App\System\Foundation\SystemQuery\SystemQueryScope;
use App\System\IsbnApplication\Database\Models\IsbnApplication;
use App\System\Publisher\Database\Models\Publisher;
use App\System\User\Database\Models\ActiveUser;
use Illuminate\Http\Request;

/***
 * Class NicoAuth
 * @package App\Http\Middleware
 */
class NicoAuth
{

    protected $namespace = "app";

    protected $namespaceId = null;

    /**
     * @var ActiveUser
     */
    protected $user;


    public function __construct()
    {
    }

    public function handle(Request $request, \Closure $next, $moduleName = null, $permissionToCheck = null, $forNamespaceId=null)
    {


        //Get the current user and load all permissions and permission Groups
        $this->user = $request->user();


        $this->user->loadPermissionAndPermissionGroups();

        //If the user is a root user, he has unlimited access.
        if ($this->user->isRoot() || !$moduleName) {
            return $next($request);
        }

        //Set the global Scope query
        $this->setGlobalScopeQueries();

        if($permissionToCheck !=null) {
            // we don't need to check other.
           if($this->user->can($permissionToCheck)) {
               return $next($request);
           }else {
               throw new ForbiddenException('You do not have permission to perform that action', AppConstants::ERR_NOT_SUFFICIENT_PERMISSION);
           }
        }



        //Determine permission needed by the request
        $permission = $this->getRequiredPermissionForRequest($request, $moduleName);

        if ($permissionToCheck) {
            $permission = $this->namespace . "." . $permissionToCheck;
        }
        if ($this->user->can($permission, $this->namespaceId)) {
            return $next($request);
        }


        throw new ForbiddenException('You do not have permission to perform that action', AppConstants::ERR_NOT_SUFFICIENT_PERMISSION);
    }

    /**
     * Resolve the namespace
     * @param Request $request
     */

    /**
     * @param Request $request
     * @param $moduleName
     * @param $permissionName
     * @return mixed
     */
    protected function getRequiredPermissionForRequest(Request $request, $moduleName = null)
    {
        $verb = strtolower($request->getMethod());
        $permission = "";
        switch ($verb) {
            case 'get':
            case 'options':
                $permission = "{$moduleName}.read";
                break;
            case 'post':
                $permission = "{$moduleName}.create";
                break;
            case 'put':
            case 'patch':
                $permission = "{$moduleName}.update";;
                break;
            case 'delete':
                $permission = "{$moduleName}.delete";;
                break;
        }
        return $this->namespace . "." . $permission;
    }

    /**
     * Set global scope queries
     */
    protected function setGlobalScopeQueries()
    {

        if ($this->user->isA('app.registered_users')) {

            $query = SystemQueryScope::getQueryInstance('users');
            $query->addWhere('id','=', $this->user->id);
            // publishers
            $query = SystemQueryScope::getQueryInstance('publishers');
            $query->addWhere('owner','=', $this->user->id);
            $publisherIds = Publisher::where('owner', $this->user->id)->pluck('id');

            // applications
            $query = SystemQueryScope::getQueryInstance('isbn_applications');
            $query->addWhereClosure(function ($q) use ($publisherIds) {
                $q->whereIn('isbn_applications.publisher_id', $publisherIds->all());
            });

            // issued isbns
            $query = SystemQueryScope::getQueryInstance('issued_isbns');
            $query->addWhereIn("issued_isbns.publisher_id", $publisherIds->all());
            // prefixes
            $query = SystemQueryScope::getQueryInstance('publisher_prefixes');
            $query->addWhereIn('publisher_prefixes.publisher_id', $publisherIds->all());


        }

    }

}
