<?php
/**
 * Created by PhpStorm.
 * User: Amar
 * Date: 2/1/2017
 * Time: 1:40 AM
 */

namespace App\Http\Controllers\Auth;


use App\Events\User\BeforeUserResponse;
use App\Events\User\UserLoginLogout;
use App\Http\Controllers\Controller;
use App\Http\Requests\UserChangePasswordRequest;
use App\System\ActivityLog\ActivityLogAction;
use App\System\AppConstants;
use App\System\Foundation\FileUpload\Upload;
use App\System\Site\Database\Models\Reseller;
use App\System\Test;
use App\System\User\Database\Models\User;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use NicoSystem\Exceptions\NicoAuthenticationException;
use NicoSystem\Foundation\NicoResponseTraits;

class AuthController extends Controller
{

    use NicoResponseTraits;
    use Upload;

    /**
     * @param $request
     *
     * @return mixed
     */
    protected function getAuthUserOrFail($request)
    {
        $user = $request->user();
        if (!$user) {
            throw new NicoAuthenticationException("User is not logged in or token is not provided.", AppConstants::ERR_USER_NOT_LOGGED_IN_OR_TOKEN_ABSENT);
        }
        event(new BeforeUserResponse($user));
        return $user;
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function getAuthUser(Request $request)
    {
        $user = $this->getAuthUserOrFail($request);
        return $this->responseOk($user);
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function logout(Request $request)
    {
        $user = $this->getAuthUserOrFail($request);
        event(new UserLoginLogout('logout'));
        $user->token()->revoke();
        return $this->responseOk(null);
    }

    /**
     * Update profile pic
     * @param Request $request
     * @return mixed
     * @throws \Illuminate\Validation\ValidationException
     */
    public function updateAvatar(Request $request)
    {
        $rules = 'required|mimes:jpg,jpeg,png|max:1024';
        $path = config('fileupload.image_directory');
        if (config('fileupload.create_folder_for_user')) {
            $path = $path . "/" . Auth::user()->id;
        }
        $result = $this->upload($request, $path, true, $rules, 'image');
        $user = Auth::user();
        $user->image_url = array_get($result, 'path');
        $user->save();
        return $this->responseOk($user);
    }

    /**
     * @param UserChangePasswordRequest $request
     * @return mixed
     */
    public function updatePassword(UserChangePasswordRequest $request)
    {

        $user = Auth::user();
        $user->password = bcrypt($request->new_password);
        $user->is_locked = User::STATUS_UNLOCKED;
        $user->save();
        return $this->responseOk($user, 'Password changed successfully');
    }

    /**
     * @param Request $request
     * @return mixed
     */
    public function update(Request $request)
    {
        $request->validate(['first_name' => 'required|max:40',
            'last_name' => 'required|max:40',
            'middle_name' => 'nullable|max:40',
            'username' => 'nullable|unique:users,username,'.Auth::user()->id,
            ]);

        $data=$request->only('first_name', 'last_name', 'username', 'middle_name','image_url');
        $user = Auth::user();
        $user->fill($data);
        $user->save();
        event(new BeforeUserResponse($user));
        return $this->responseOk($user, 'Profile Updated successfully');

    }

    /**
     * @param Request $request
     * @return mixed
     */
    public function permissions(Request $request)
    {
        $user=Auth::user();
        $permission=$user->permissionGroups()->select('key as group','permission')->get();
        return $this->responseOk($permission);
    }
}