<?php
/**
 * Created by PhpStorm.
 * User: Amar
 * Date: 1/19/2017
 * Time: 9:46 PM
 */

namespace App\Http\Controllers\Auth;

use App\Events\User\BeforeUserResponse;
use App\Events\User\UserLoginLogout;
use App\System\ActivityLog\ActivityLogAction;
use App\System\AppConstants;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Laravel\Passport\Http\Controllers\AccessTokenController;
use NicoSystem\Exceptions\NicoBadRequestException;
use NicoSystem\Foundation\NicoResponseTraits;
use Psr\Http\Message\ServerRequestInterface;

/**
 * Class ApiLoginController
 * @package App\Http\Controllers\Auth
 */
class ApiLoginController extends AccessTokenController
{
    use NicoResponseTraits,AuthenticatesUsers;

    const GRANT_TYPE_PASSWORD = 'password';

    const GRANT_TYPE_REFRESH_TOKEN = 'refresh_token';




    /**
     * @param Response $response
     * @return mixed
     */
    protected function parseResultForResponse(Response $response)
    {
        if ($response->getStatusCode() == 401) {
            //invalid username password issue
            $message = json_decode($response->getContent());

            if ($message->error == AppConstants::ERR_INVALID_CREDENTIAL ||
                $message->error==AppConstants::ERR_TOO_MANY_LOGIN_ATTEMPT ) {
                return $this->responseUnAuthorize($message->message, $message->error);
            } else {
                return $this->responseBadRequest($message->message, $message->error);
            }

        } elseif ($response->getStatusCode() == 200) {
            $user = Auth::user();

            event(new BeforeUserResponse($user));
            event(new UserLoginLogout('login'));
            $content = json_decode($response->getContent());
            //TODO: Find out why headers are not being set. and then remove these pieces of codes.
            $user->refresh_token = $content->refresh_token;
            $user->access_token = $content->access_token;
            $user->token_type = $content->token_type;
            $user->expires_in = $content->expires_in;
            return $this->responseOk($user, 'ok', 'ok', ['Access-Token' => $content->access_token,
                'Refresh-Token' => $content->refresh_token,
                'Token-Type' => $content->token_type,
                'Expires-In' => $content->expires_in,
            ]);
        }

        return $this->responseServerError($response->getContent());
    }

    /**
     * @param $grantType
     * @return array
     */
    protected
    function getAuthParamsForClient($grantType)
    {
        return [
            'grant_type' => $grantType,
            'client_id' => config('baseclient.id'),
            'client_secret' => config('baseclient.secret')
        ];
    }

    protected function failIfConfirmationFails ($user) {
        if($user->emailIsConfirmed()){
            return true;
        }
        throw new NicoBadRequestException("E-mail confirmation is required",'err_email_confirmation');
    }
    
    /**
     *Authenticate User with check multiple Login attempts
     *
     * @param ServerRequestInterface $request
     * @return mixed
     */
    public function authenticate(ServerRequestInterface $request)
    {

        //validate input
        $rule = ['username' => 'required', 'password' => 'required'];
        $validator = Validator::make($request->getParsedBody(), $rule);
        if ($validator->fails()) {
            return $this->responseValidationError($validator->messages(), 'Username and password is required', AppConstants::ERR_FORM_VALIDATION);
        }

        //convert PSR-7 Request  to Laravel (Symfony Request)
        $httpRequest=new Request($request->getParsedBody());

        //check if user has reached max Login attempts
        if($this->hasTooManyLoginAttempts($httpRequest))
        {
            $this->fireLockoutEvent($httpRequest);

            return $this->responseTooManyAttempts('User account is disable due to too many wrong attempt', AppConstants::ERR_TOO_MANY_LOGIN_ATTEMPT);
        }

        $credentials=[
            'email'=>$httpRequest->username,
            'password'=>$httpRequest->password,
        ];

        if (Auth::attempt($credentials))
        {

            //Authentication passed...
            //reset failed login attemps
            $this->clearLoginAttempts($httpRequest);
            $this->failIfConfirmationFails(Auth::user());

            $request = $request->withParsedBody(
                array_merge(
                    $request->getParsedBody(),
                    $this->getAuthParamsForClient(static::GRANT_TYPE_PASSWORD))
            );
            $response = $this->issueToken($request);
            return $this->parseResultForResponse($response);
        }
        else
        {
            //count user failed login attempts
            $this->incrementLoginAttempts($httpRequest);

            return $this->responseUnAuthorize('The user credentials were incorrect.', AppConstants::ERR_INVALID_CREDENTIAL);
        }

    }

    /**
     * Refresh the token
     * @param ServerRequestInterface $request
     * @return mixed
     */
    public function refreshToken(ServerRequestInterface $request)
    {
        $rule = ['refresh_token' => 'required'];

        $validator = Validator::make($request->getParsedBody(), $rule);
        if ($validator->fails()) {
            return $this->responseValidationError($validator->messages(), 'Refresh token is required', AppConstants::ERR_FORM_VALIDATION);
        }
        $request = $request->withParsedBody(array_merge($request->getParsedBody(), $this->getAuthParamsForClient(static::GRANT_TYPE_REFRESH_TOKEN)));
        $response = $this->issueToken($request);
        return $this->parseResultForResponse($response);
    }


}
