<?php

namespace App\Exceptions;

use App\System\AppConstants;
use App\System\ErrorLog\Logger;
use Exception;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Debug\Dumper;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\ValidationException;
use NicoSystem\Exceptions\NicoException;
use NicoSystem\Foundation\NicoResponseTraits;
use Symfony\Component\HttpKernel\Exception\HttpException;

class Handler extends ExceptionHandler
{
    use NicoResponseTraits;
    /**
     * A list of the exception types that are not reported.
     *
     * @var array
     */
    protected $dontReport = [
        //
    ];

    /**
     * A list of the inputs that are never flashed for validation exceptions.
     *
     * @var array
     */
    protected $dontFlash = [
        'password',
        'password_confirmation',
    ];

    /**
     * Test if the request is an API call
     * @param Request $request
     * @return boolean
     */
    protected function isApiRequest(Request $request){
        return is_api_request($request);
    }

    /**
     * Send Exception error Report
     * @param Exception $exception
     */
    protected function sendErrorReportViaMail(Exception $exception){
        $content = $this->prepareResponse(request(),$exception);
        try{
            Mail::send('errors.developer-mail',compact('content'),function($message)  {

                $message->to(config('mail.developer.email'),config('mail.developer.name'));
                $subject = config('app.name').": Error From " . env("APP_ENV");
                $message->subject($subject);
            });
        }catch (\Exception $exception){
            //Do nothing if error mail is not sent because it is not compulsory
        }

    }

    /**
     * Report or log an exception.
     *
     * This is a great spot to send exceptions to Sentry, Bugsnag, etc.
     *
     * @param  \Exception  $exception
     * @return void
     */
    public function report(Exception $exception)
    {
        if($exception->getCode()==500 && config('app.send_error_mail_to_developer',false)){
            $this->sendErrorReportViaMail($exception);
        }
        parent::report($exception);
    }

    /**
     * Render an exception into an HTTP response.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Exception  $exception
     * @return \Illuminate\Http\Response|mixed
     */
    public function render($request, Exception $exception)
    {

        if($this->isApiRequest($request)){
            if($request->getMethod()=='OPTIONS'&& $exception instanceof HttpException){
                return new Response();
            }

            if($exception instanceof  NicoException){
                return $this->nicoResponse($exception->getResponseBody(),$exception->getCode(),$exception->getMessage(),$exception->getResponseCode());
            }elseif($exception instanceof AuthenticationException){
                return $this->responseUnAuthorize($exception->getMessage());
            }elseif($exception instanceof \Swift_TransportException){
                return $this->responseServerError("E-mail couldn't be sent due to authorization error. Please check smtp configuration",AppConstants::ERR_MAIL_SERVER_AUTH_OR_CONF);
            }elseif($exception instanceof ValidationException){
                return $this->responseValidationError($exception->errors());
            }elseif ($exception instanceof ModelNotFoundException){
                return $this->responseNotFound();
            }elseif($exception instanceof HttpException){
                return $this->nicoResponse('',$exception->getStatusCode(),$exception->getMessage(),AppConstants::getAppMsgCodeFromStatusCode($exception->getStatusCode()));
            }

            return $this->nicoResponse(null,$exception->getCode(),$exception->getMessage()." ".$exception->getFile().": line ".$exception->getLine(),$exception->getCode());
        }

        return parent::render($request, $exception);
    }
}
