<?php
/**
 * Created by PhpStorm.
 * User: Amar
 * Date: 12/30/2016
 * Time: 11:07 PM
 */

namespace NicoSystem\Repositories;


use App\System\AppBaseModel;
use App\System\AppConstants;
use App\System\Foundation\SystemQuery\SystemQueryScope;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Event;
use NicoSystem\Data\Status;
use NicoSystem\Exceptions\NicoBadRequestException;
use NicoSystem\Exceptions\NicoException;
use NicoSystem\Filters\BaseFilter;
use NicoSystem\Foundation\Database\BaseModel;
use NicoSystem\Interfaces\BasicCrudInterface;

abstract class BaseRepository implements BasicCrudInterface
{
    /**
     * @var BaseModel
     */
    protected $model = null;

    protected $events = [];

    protected $query = null;

    public function __construct(BaseModel $model)
    {
        $this->model = $model;
    }


    /**
     * @param array $params
     * @return \Illuminate\Database\Eloquent\Builder
     */
    protected function attachOrderByQuery($params = [])
    {
        $builder = $this->getQuery();
        $orderColumn = array_get($params, 'sort_by', $this->model->defaultSortColumn());
        $orderBy = array_get($params, 'sort_order', $this->model->defaultSortOrder());

        if (!in_array($orderColumn, $this->model->sortableColumns())) {
            $orderColumn = $this->model->defaultSortColumn();
        }

        $orderColumn = $this->model->mapSortKey($orderColumn);

        if (!$orderBy || !in_array($orderBy, ['asc', 'desc'])) {
            $orderBy = $this->model->defaultSortOrder();
        }

        if ($orderColumn) {

            $builder->orderBy($orderColumn, $orderBy);
        }
        return $builder;
    }

    /**
     * @param $id
     * @param array $attributes
     * @return BaseModel
     */
    public function getById($id, array $attributes = [])
    {
        return $this->getQuery()->findOrFail($id);
    }

    /**
     * @param Builder $builder
     * @return BaseFilter
     */
    public abstract function getFilter(Builder $builder);

    /**
     * @param array $params
     * @param bool $paginate
     * @param array $attributes
     * @param array $withAggregate
     * @return mixed
     */
    public function getList(array $params = [], $paginate = true, array $attributes = [])
    {
        $builder = $this->attachOrderByQuery($params);
        $filter = $this->getFilter($builder);

        if ($filter) {
            $filter->attachFilterQuery($params);
        }

        if(array_get($params,'all') == 1) {
            $paginate = false;
        }

        $this->onBeforeResult($builder);

        if ($attributes) {
            $builder->select($attributes);
        }
	
	if ($paginate) {
            return $builder->paginate();
        }

        return $builder->get();
    }

    public function onBeforeResult(Builder $builder)
    {

    }

    protected function beforeSavingModel (BaseModel $model,  array $attributes = []) {

    }

    /**
     * @param $id
     * @param array $options
     * @return mixed
     * @throws \Exception
     */
    public function destroy($id, array $options = [])
    {
        $model = $this->getQuery()->findOrFail($id);
        $this->dispatchEvent('deleting', $model);
        $model->delete();
        $this->dispatchEvent('deleted', $model);
        return true;
    }

    protected function dispatchCreatingUpdatingEvent ($new, $model) {
        if (!$new) {
            $this->dispatchEvent('updating', $model);
        } else {
            $this->dispatchEvent('creating', $model);
        }
    }
    protected function dispatchCreatedUpdatedEvent ($new, $model) {
        if (!$new) {
            $this->dispatchEvent('created', $model);
        } else {
            $this->dispatchEvent('updated', $model);
        }
    }

    /**
     * @param array $attributes
     * @param null $id
     * @return mixed
     */
    protected function save(array $attributes, $id = null)
    {
        if ($id) {
            $model = $this->getQuery()->findOrFail($id);
        } else {
            $model = $this->model->newInstance();
        }
        $model->fill($attributes);
        $this->beforeSavingModel($model, $attributes);
        $this->dispatchCreatingUpdatingEvent($id!=null, $model);
        $model->save();
        $this->dispatchCreatedUpdatedEvent($id !=null, $model);
        return $model;
    }

    /**
     * @param array $inputs
     * @return mixed
     */
    public function create(array $inputs)
    {
        return $this->save($inputs);
    }

    /**
     * @param $id
     * @param array $attributes
     * @return mixed
     */
    public function update($id, array $attributes)
    {
        return $this->save($attributes, $id);
    }

    /**
     * @param $id
     * @return AppBaseModel|BaseModel
     */
    public function togglePublish($id)
    {
        if (is_numeric($id)) {
            $model = $this->getById($id);
        } elseif ($id instanceof AppBaseModel) {
            $model = $id;
        } else {
            throw new NicoException("", 500);
        }
        if ($model->status == Status::STATUS_UNPUBLISHED) {
            $model->status = Status::STATUS_PUBLISHED;
        } elseif ($model->status == Status::STATUS_PUBLISHED) {
            $model->status = Status::STATUS_UNPUBLISHED;
        } elseif ($model->status == Status::STATUS_SUSPENDED) {
            throw new NicoBadRequestException("Cannot modify the model because the status is suspended", AppConstants::ERR_MODEL_NOT_EDITABLE);
        }
        $this->dispatchEvent('updating', $model);
        $model->save();
        $this->dispatchEvent('updated', $model);
        return $model;
    }

    /**
     * @param $eventName
     * @param $model
     * @param $user
     */
    protected function dispatchEvent($eventName, $model)
    {
        if (array_key_exists($eventName, $this->events)) {
            Event::dispatch(new $this->events[$eventName]($model, Auth::user()));
        }
    }

    /**
     * @param $id
     * @param $field
     * @param $value
     * @return mixed
     */
    protected function updateSingle($id, $field, $value)
    {
        $model = $this->getQuery()->findOrFail($id);
        $model->$field = $value;
        $this->dispatchEvent('updating', $model);
        $model->save();
        $this->dispatchEvent('updated', $model);
        return $model;
    }

    protected function getQuery($new = false)
    {
        if(!$this->query || $new === true) {
            $this->query = $this->model->newQuery();
        }
        return $this->query;
    }

}
