<?php
/**
 * Created by PhpStorm.
 * User: Amar
 * Date: 12/30/2016
 * Time: 11:04 PM
 */

namespace NicoSystem\Foundation\Database;


use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Support\Str;
use NicoSystem\Data\Status;
use NicoSystem\Exceptions\DataAssertionException;
use NicoSystem\Foundation\Database\Relation\HasManyExtended;

abstract class BaseModel extends Model
{
    /**
     *  True to make repsonses to snakecase
     * @var bool
     */
    protected $snakecase = true;
    /**
     * The phone attributes
     * @var array
     */
    protected $phones = [];

    /**
     * @var string
     */
    protected $defaultSortColumn = "created_at";
    /**
     * @var string
     */
    protected $defaultSortOrder = "asc";
    /**
     * @var array
     */
    protected $sortableColumns = ['title','created_at','id'];
    /**
     * @var array
     */
    protected $sortKeyMaps = [];

    /**
     * @return array
     */
    public function sortableColumns(){
        return $this->sortableColumns;
    }

    /**
     * @return string
     */
    public function defaultSortColumn(){
        return $this->defaultSortColumn;
    }

    /**
     * @return string
     */
    public function defaultSortOrder(){
        return $this->defaultSortOrder;
    }

    /**
     * @param $key
     * @return bool|null
     */
    public function mapSortKey($key) {
        return isset($this->sortKeyMaps[$key]) ? $this->sortKeyMaps[$key] : $key;
    }

    /**
     * Append ascending order in the query
     * @param Builder $query
     * @param $colname
     * @param bool $asc
     */
    public function scopeAscending(Builder $query, $colname,$asc = true){
        $order = 'asc';
        if($asc===false){
            $order = 'desc';
        }
        $query->orderBy($colname,$order);
    }

    /**
     * Append descending order in the query
     * @param Builder $query
     * @param $colname
     */
    public function scopeDescending(Builder $query, $colname ){
        $this->scopeAscending($query,$colname,false);
    }

    /**
     * Add a publish check to query
     * @param Builder $query
     * @param string $colName
     */
    public function scopePublished(Builder $query,$colName='status'){
        $query->where($colName,Status::STATUS_PUBLISHED);
    }

    /**
     * @param Builder $query
     * @param string $colName
     */
    public function scopeUnpublished(Builder $query, $colName='status'){
        $query->where($colName,Status::STATUS_UNPUBLISHED);
    }

    /**
     * @param $related
     * @param \Closure|null $closure
     * @param null $foreignKey
     * @param null $localKey
     * @return HasOne
     */
    public function hasOneExtended($related, \Closure $closure = null, $foreignKey = null, $localKey = null)
    {
        $foreignKey = $foreignKey ?: $this->getForeignKey();

        $instance = new $related;

        $localKey = $localKey ?: $this->getKeyName();

        $query  = $instance->newQuery();

        if($closure!==null){
            $closure($query);
        }

        return new HasOne($query, $this, $instance->getTable().'.'.$foreignKey, $localKey);
    }

    /**
     * @param $related
     * @param \Closure|null $closure
     * @param null $foreignKey
     * @param null $localKey
     * @return HasManyExtended
     */
    public function hasManyExtended($related, \Closure $closure= null , $foreignKey = null, $localKey = null)
    {
        $foreignKey = $foreignKey ?: $this->getForeignKey();

        $instance = new $related;

        $localKey = $localKey ?: $this->getKeyName();

        $query = $instance->newQuery();

        if($closure!==null){
            $closure($query);
        }

        return new HasManyExtended($query, $this, $instance->getTable().'.'.$foreignKey, $localKey);
    }

    /**
     * Determine if an assertion exists for an attribute.
     *
     * @param  string  $key
     * @return bool
     */
    public function hasAssertion($key)
    {
        return method_exists($this, 'assert'.Str::studly($key).'Attribute');
    }

    /**
     * Override the set attribute to assert the data
     * @override
     * @param string $key
     * @param mixed $value
     * @throws DataAssertionException
     * @return Model
     */
    public function setAttribute($key, $value)
    {
        if ($this->hasAssertion($key)) {
            $method = 'assert'.Str::studly($key).'Attribute';
            if($this->{$method}($value)!==true){
                throw (new DataAssertionException())->setModel($this)->setDataName($key);
            }
        }

       return parent::setAttribute($key, $value);
    }

    /**
     * Override method
     * @return array
     */
    public function toArray()
    {
        if($this->snakecase===true){
            return parent::toArray();
        }

        $items = parent::toArray();
        $returns = [];
        foreach ($items as $key=>$value){
            $returns[camel_case($key)]  = $value;
        }
        return $returns;
    }

}
